/* Copyright (C) 2003-2007  The Chemistry Development Kit (CDK) project
 *
 * Contact: cdk-devel@lists.sourceforge.net
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public License
 * as published by the Free Software Foundation; either version 2.1
 * of the License, or (at your option) any later version.
 * All we ask is that proper credit is given for our work, which includes
 * - but is not limited to - adding the above copyright notice to the beginning
 * of your source code files, and to any copyright notice that you may distribute
 * with programs based on this work.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 */
package org.openscience.cdk.dict;

import java.io.IOException;
import java.io.Reader;
import java.util.Hashtable;
import java.util.Iterator;
import java.util.Map;

import org.openscience.cdk.tools.ILoggingTool;
import org.openscience.cdk.tools.LoggingToolFactory;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;

import javax.xml.parsers.SAXParserFactory;


/**
 * Dictionary with entries.
 *
 * <p>FIXME: this should be replaced by an uptodate Dictionary Schema
 * DOM type thing.
 *
 * @author     Egon Willighagen
 * @cdk.githash
 * @cdk.created    2003-08-23
 * @cdk.keyword    dictionary
 * @cdk.module     dict
 */
public class Dictionary {

    private final Map<String, Entry> entries;
    private String             ownNS = null;

    public Dictionary() {
        entries = new Hashtable<>();
    }

    public static Dictionary unmarshal(Reader reader) {
        ILoggingTool logger = LoggingToolFactory.createLoggingTool(Dictionary.class);
        DictionaryHandler handler = new DictionaryHandler();
        XMLReader parser = null;
        try {
            SAXParserFactory factor = SAXParserFactory.newInstance();
            factor.setNamespaceAware(true);
            factor.setFeature("http://xml.org/sax/features/external-general-entities", false);
            parser = factor.newSAXParser().getXMLReader();
            logger.debug("Using " + parser);
        } catch (Exception e) {
            logger.error("Could not instantiate any JAXP parser!");
            logger.debug(e);
        }

        try {
            if (parser == null) {
                logger.debug("parser object was null!");
                return null;
            }
            parser.setFeature("http://xml.org/sax/features/validation", false);

            logger.debug("Deactivated validation");
        } catch (SAXException e) {
            logger.warn("Cannot deactivate validation.");
            logger.debug(e);
        }
        parser.setContentHandler(handler);
        Dictionary dict = null;
        try {
            parser.parse(new InputSource(reader));
            dict = handler.getDictionary();
        } catch (IOException e) {
            logger.error("IOException: " + e);
            logger.debug(e);
        } catch (SAXException saxe) {
            logger.error("SAXException: " + saxe.getClass().getName());
            logger.debug(saxe);
        }
        return dict;
    }

    void addEntry(Entry entry) {
        entries.put(entry.getID().toLowerCase(), entry);
    }

    public Entry[] getEntries() {
        int size = entries.size();
        Entry[] entryArray = new Entry[size];
        Iterator<Entry> elements = entries.values().iterator();
        int counter = 0;
        while (elements.hasNext() && counter < size) {
            entryArray[counter] = elements.next();
            counter++;
        }
        return entryArray;
    }

    public boolean hasEntry(String identifier) {
        return entries.containsKey(identifier);
    }

    public Entry getEntry(String identifier) {
        return entries.get(identifier);
    }

    public int size() {
        return entries.size();
    }

    public void setNS(String nameSpace) {
        ownNS = nameSpace;
    }

    public String getNS() {
        return ownNS;
    }
}
