/* $Id: hdrl_overscan.h,v 1.3 2013-10-16 17:48:58 cgarcia Exp $
 *
 * This file is part of the HDRL
 * Copyright (C) 2012,2013 European Southern Observatory
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

/*
 * $Author: cgarcia $
 * $Date: 2013-10-16 17:48:58 $
 * $Revision: 1.3 $
 * $Name: not supported by cvs2svn $
 */

#ifndef HDRL_OVERSCAN_H
#define HDRL_OVERSCAN_H

/*-----------------------------------------------------------------------------
                                Include
 -----------------------------------------------------------------------------*/

#include <hdrl_utils.h>

#include <cpl.h>

/*-----------------------------------------------------------------------------
                                Define
 -----------------------------------------------------------------------------*/

#define HDRL_OVERSCAN_FULL_BOX -1

/*----------------------------------------------------------------------------*/
/**
  @struct hdrl_overscan_compute_result_s
  @brief Contains the Overscan Computation results
 */
/*----------------------------------------------------------------------------*/
struct hdrl_overscan_compute_result_s
{
    /** The direction in which the image were collapsed in the computation */
    hdrl_direction      correction_direction;     
    /** The overscan correction as a 1D double image */
    cpl_image       *   correction;
    /** The error as a 1D double image */
    cpl_image       *   error;
    /** The number of good pixels that contributed as a 1D double image */
    cpl_image       *   contribution;
    /** The \f$\chi^{2}\f$ as a 1D double image */
    cpl_image       *   chi2;
    /** The reduced \f$\chi^{2}\f$ as a 1D double image */
    cpl_image       *   red_chi2;
    /** The low threshold below which the pixels are rejected 
        as a 1D double image. Only for sigma-clipping collapsing method */
    cpl_image       *   sigclip_reject_low;
    /** The high threshold above which the pixels are rejected 
        as a 1D double image. Only for sigma-clipping collapsing method */
    cpl_image       *   sigclip_reject_high;
};
typedef struct hdrl_overscan_compute_result_s hdrl_overscan_compute_result_t;

/*----------------------------------------------------------------------------*/
/**
  @struct hdrl_overscan_correct_result_s
  @brief Contains the Overscan Correction results
 */
/*----------------------------------------------------------------------------*/
struct hdrl_overscan_correct_result_s
{
    /** The Overscan corrected image */
    cpl_image   *   ima_corrected;
    /** The propagated error */
    cpl_image   *   error;
    /** Pixels marked as bad by algorithm, encoded by user choice */
    cpl_image   *   badmask;
};
typedef struct hdrl_overscan_correct_result_s hdrl_overscan_correct_result_t;

/*-----------------------------------------------------------------------------
                            Functions prototypes
 -----------------------------------------------------------------------------*/

/*----------------------------------------------------------------------------
                           Overscan Parameters
  ----------------------------------------------------------------------------*/
CPL_BEGIN_DECLS

hdrl_parameter * hdrl_overscan_parameter_create(hdrl_direction, double, int, 
        hdrl_parameter *, hdrl_parameter *) ;
void hdrl_overscan_parameter_destroy(void *) ;
cpl_error_code hdrl_overscan_parameter_update(hdrl_parameter *, hdrl_direction,
        double, int) ;
cpl_boolean hdrl_overscan_parameter_check(const hdrl_parameter *) ;
hdrl_direction hdrl_overscan_parameter_get_correction_direction(
        const hdrl_parameter *) ;
double hdrl_overscan_parameter_get_ccd_ron(
        const hdrl_parameter *) ;
int hdrl_overscan_parameter_get_box_hsize(
        const hdrl_parameter *) ;
hdrl_parameter * hdrl_overscan_parameter_get_collapse(
        const hdrl_parameter *) ;
hdrl_parameter * hdrl_overscan_parameter_get_rect_region(
        const hdrl_parameter *) ;
cpl_error_code hdrl_overscan_parameter_verify(const hdrl_parameter *,
        cpl_size, cpl_size) ;

hdrl_parameter * hdrl_overscan_parameter_parse_parlist(
        cpl_parameterlist *, const char *) ;
cpl_parameterlist * hdrl_overscan_parameter_create_parlist(const char *, 
        const char *, const char *, const char *, int, double,
        hdrl_parameter *, const char *, hdrl_parameter *) ;

/*----------------------------------------------------------------------------
                           Overscan Computation
  ----------------------------------------------------------------------------*/
hdrl_overscan_compute_result_t * hdrl_overscan_compute(
        const cpl_image         *   source,
        const hdrl_parameter    *   params) ;
void hdrl_overscan_compute_result_destroy(hdrl_overscan_compute_result_t *) ;

/*----------------------------------------------------------------------------
                           Overscan Correction
  ----------------------------------------------------------------------------*/
hdrl_overscan_correct_result_t * hdrl_overscan_correct(
        const cpl_image                         *   source,
        const cpl_image                         *   source_error,
        const hdrl_parameter                    *   region,
        const hdrl_bitmask_t                        reject_code,
        const hdrl_overscan_compute_result_t    *   os_computation) ;
void hdrl_overscan_correct_result_destroy(hdrl_overscan_correct_result_t *) ;
CPL_END_DECLS

#endif
