/*
 * Diagnostics - a unified framework for code annotation, logging,
 * program monitoring, and unit-testing.
 *
 * Copyright (C) 2009 Christian Schallhart <christian@schallhart.net>,
 *                    Michael Tautschnig <tautschnig@forsyte.de>
 *               2008 model.in.tum.de group, FORSYTE group
 *               2006-2007 model.in.tum.de group
 *               2002-2005 Christian Schallhart
 *  
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
 */


#ifndef DIAGNOSTICS__EXTENSIONS__MEMORY__MALLOC_ALLOCATOR__INCLUDE_GUARD
#define DIAGNOSTICS__EXTENSIONS__MEMORY__MALLOC_ALLOCATOR__INCLUDE_GUARD

#include <diagnostics/extensions/memory/namespace.hpp>

// malloc/free
#include <cstdlib>

// ::std::allocator
#include <memory>

// ::std::numeric_limits
#include <limits>

DIAGNOSTICS_NAMESPACE_BEGIN;
MEMORY_NAMESAPCE_BEGIN;
INTERNAL_NAMESPACE_BEGIN;

/**
 * @class Malloc_Allocator
 *
 * @brief It is a straight forward STL allocator which uses internally
 * malloc/free for its memory management. We use this allocator to be
 * independent from new/delete allocations -- and more importantly: we
 * must be independent of STL internal pooling and mutexing. 
 *
 * As an example of the problems, that could occur: 
 *
 * @arg STL container locks the global memory pool.
 * @arg STL container requires memory beyond the available pool size
 * @arg .. pool calls new
 * @arg .. .. Allocation_Database_Implementation has to register the allocation
 * @arg .. .. .. its underlying STL container have to allocate memory
 * @arg .. .. .. .. attempts to _lock_ the global memory pool (DEADLOCK)
 *
 * Thus we use this alloctor for internal containers. 
 *
 * @nosubgrouping
 */

template<typename T>
class Malloc_Allocator 
{
public: 
    typedef T value_type;
    typedef value_type* pointer;
    typedef const value_type* const_pointer;
    typedef value_type& reference;
    typedef const value_type& const_reference;
    typedef ::std::size_t size_type;
    typedef ::std::ptrdiff_t difference_type;
	
private:
	typedef typename ::std::allocator<void>::const_pointer  const_void_pointer;

public : 
    template<typename U>
    struct rebind { typedef Malloc_Allocator<U> other;};

public : 
    inline Malloc_Allocator() {}
    inline ~Malloc_Allocator() {}
    inline Malloc_Allocator(Malloc_Allocator const&) {}
    template<typename U>
    inline Malloc_Allocator(Malloc_Allocator<U> const&) {}

    inline pointer address(reference r) { return &r; }
    inline const_pointer address(const_reference r) { return &r; }

    inline pointer allocate(size_type cnt, const_void_pointer = 0) 
	{ 
		return reinterpret_cast<pointer>(::std::malloc(cnt * sizeof (T))); 
    }
    inline void deallocate(pointer p, size_type) 
	{ 
        ::std::free(p); 
    }

    inline size_type max_size() const 
	{ 
        return ::std::numeric_limits<size_type>::max() / sizeof(T);
	}

    inline void construct(pointer p, const T& t) { new(p) T(t); }
    inline void destroy(pointer p) { p->~T(); }
	
    inline bool operator==(Malloc_Allocator const&) { return true; }
    inline bool operator!=(Malloc_Allocator const& a) { return !operator==(a); }
};  


INTERNAL_NAMESPACE_END;
MEMORY_NAMESAPCE_END;
DIAGNOSTICS_NAMESPACE_END;

#endif

// vim:ts=4:sw=4
