/**
 * Copyright (C) 2002, 2003 Billy Biggs <vektor@dumbterm.net>.
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * "Software"), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <png.h>
#include "pngoutput.h"

struct pngoutput_s
{
    FILE *f;
    png_structp png_ptr;
    png_infop info_ptr;
    int width, height;
};

static void pngoutput_header_8bit( pngoutput_t *pngoutput, int alpha )
{
    png_set_sRGB_gAMA_and_cHRM( pngoutput->png_ptr, pngoutput->info_ptr,
                                PNG_sRGB_INTENT_RELATIVE );
    png_set_IHDR( pngoutput->png_ptr, pngoutput->info_ptr,
                  pngoutput->width, pngoutput->height, 8,
                  alpha ? PNG_COLOR_TYPE_RGB_ALPHA : PNG_COLOR_TYPE_RGB,
                  PNG_INTERLACE_NONE, PNG_COMPRESSION_TYPE_DEFAULT,
                  PNG_FILTER_TYPE_DEFAULT );
    png_write_info( pngoutput->png_ptr, pngoutput->info_ptr );
}

pngoutput_t *pngoutput_new( const char *filename,
                            int width, int height,
                            int alpha, int compression )
{
    pngoutput_t *pngoutput = malloc( sizeof( pngoutput_t ) );

    if( !pngoutput ) return 0;

    pngoutput->f = 0;
    pngoutput->png_ptr = 0;
    pngoutput->info_ptr = 0;
    pngoutput->width = width;
    pngoutput->height = height;

    pngoutput->f = fopen( filename, "w" );
    if( !pngoutput->f ) {
        fprintf( stderr, "pngoutput: Cannot open %s: %s\n",
                 filename, strerror( errno ) );
        pngoutput_delete( pngoutput );
        return 0;
    }

    pngoutput->png_ptr = png_create_write_struct( PNG_LIBPNG_VER_STRING, 0, 0, 0 );
    if( !pngoutput->png_ptr ) {
        fprintf( stderr, "pngoutput: Cannot open PNG write struct.\n" );
        pngoutput_delete( pngoutput );
        return 0;
    }

    pngoutput->info_ptr = png_create_info_struct( pngoutput->png_ptr );
    if( !pngoutput->info_ptr ) {
        png_destroy_write_struct( &(pngoutput->png_ptr), 0 );
        pngoutput->png_ptr = 0;
        fprintf( stderr, "pngoutput: Cannot open PNG info struct.\n" );
        pngoutput_delete( pngoutput );
        return 0;
    }

    png_init_io( pngoutput->png_ptr, pngoutput->f );
    if( compression ) {
        png_set_filter( pngoutput->png_ptr, 0, PNG_FILTER_PAETH );
        png_set_compression_level( pngoutput->png_ptr, Z_BEST_COMPRESSION );
    } else {
        png_set_filter( pngoutput->png_ptr, 0, PNG_FILTER_NONE );
        png_set_compression_level( pngoutput->png_ptr, 0 );
    }
    pngoutput_header_8bit( pngoutput, alpha );

    return pngoutput;
}

void pngoutput_delete( pngoutput_t *pngoutput )
{
    if( pngoutput->png_ptr && pngoutput->info_ptr ) {
        png_write_end( pngoutput->png_ptr, pngoutput->info_ptr );
        png_destroy_write_struct( &pngoutput->png_ptr, &pngoutput->info_ptr );
    }
    if( pngoutput->f ) {
        fclose( pngoutput->f );
    }
    free( pngoutput );
}

void pngoutput_scanline( pngoutput_t *pngoutput, uint8_t *scanline )
{
    png_write_row( pngoutput->png_ptr, scanline );
}

