-------------------------------------------------------------------------------
--                                                                           --
--  Filename        : $Source: /cvsroot/gnade/gnade/dbi/adbc/gnu-db-adbc-driver-pgsql.adb,v $
--  Description     : Ada Database Object - Driver for MySQL                 --
--  Author          : Michael Erdmann                                        --
--  Created         : 18.1.2002                                              --
--  Last Modified By: $Author: jukano $
--  Last Modified On: $Date: 2002/02/25 20:11:28 $
--  Status          : $State: Exp $
--                                                                           --
--  Copyright (C) 2002 Michael Erdmann                                       --
--                                                                           --
--  GNADE is free software;  you can redistribute it  and/or modify it under --
--  terms of the  GNU General Public License as published  by the Free Soft- --
--  ware  Foundation;  either version 2,  or (at your option) any later ver- --
--  sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
--  OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
--  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
--  for  more details.  You should have  received  a copy of the GNU General --
--  Public License  distributed with GNAT;  see file COPYING.  If not, write --
--  to  the Free Software Foundation,  59 Temple Place - Suite 330,  Boston, --
--  MA 02111-1307, USA.                                                      --
--                                                                           --
--  As a special exception,  if other files  instantiate  generics from this --
--  unit, or you link  this unit with other files  to produce an executable, --
--  this  unit  does not  by itself cause  the resulting  executable  to  be --
--  covered  by the  GNU  General  Public  License.  This exception does not --
--  however invalidate  any other reasons why  the executable file  might be --
--  covered by the  GNU Public License.                                      --
--                                                                           --
--  This software is implemented to work with GNAT, the GNU Ada compiler.    --
--                                                                           --
--  Functional Description                                                   --
--  ======================                                                   --
--  This is the test driver for the ADO interface. It is some kind of        --
--  stub which simulates the requested funtions.                             --
--                                                                           --
--                                                                           --
--  Restrictions                                                             --
--  ============                                                             --
--  Only Linux                                                               --
--                                                                           --
--  Contact                                                                  --
--  =======                                                                  --
--  Error reports shall be handled via http://gnade.sourceforge.net          --
--  Features and ideas via: gnade-develop@lists.sourceforge.net              --
--                                                                           --
--  Author contact:                                                          --
--               purl:/net/michael.erdmann                                   --
--                                                                           --
-------------------------------------------------------------------------------
--* Ada
with System;                                use System;
with System.Storage_Elements;               use System.Storage_Elements;
with System.Address_To_Access_Conversions;

with Ada.Exceptions;                        use Ada.Exceptions;
with Ada.Tags;                              use Ada.Tags;
with Ada.Characters.Latin_1;                use Ada.Characters;
with Ada.Calendar;                          use Ada.Calendar;
with Ada.Strings.Fixed;                     use Ada.Strings.Fixed;
with Ada.Strings.Unbounded;                 use Ada.Strings.Unbounded;
with Ada.Unchecked_Conversion;

with Ada.Text_IO;                           use Ada.Text_IO;

with Interfaces.C.Strings;                  use Interfaces.C.Strings;
with Interfaces.C;                          use Interfaces.C;
use  Interfaces;

with Unchecked_Deallocation;

with GNU.DB.ADBC.Statement;                 use GNU.DB.ADBC.Statement;
with GNU.DB.ADBC.Resultset;                 use GNU.DB.ADBC.Resultset;
with GNU.DB.ADBC.Connection;                use GNU.DB.ADBC.Connection;
with GNU.DB.ADBC.Hostvariable;              use GNU.DB.ADBC.Hostvariable;
with GNU.DB.ADBC.Hostvariable.Types;        use GNU.DB.ADBC.Hostvariable.Types;

with GNU.DB.ADBC.Row;                       use GNU.DB.ADBC.Row;

package body GNU.DB.ADBC.Driver.PgSQL is
   pragma Linker_Options ("-lpq");

   Version : constant  String :=
      "$Id: gnu-db-adbc-driver-pgsql.adb,v 1.3 2002/02/25 20:11:28 jukano Exp $";

   --- ******************************************************************** ---
   --- ***  L O W    L E V E L    I N T E R F A C E   T O    P G S Q L  *** ---
   --- ******************************************************************** ---

   --- Interface to the PGSQL Library. Some of the data types are simply
   --- Addresses because the internals of these structured do not have
   --- to be known here.

   subtype PGConn is Address;
   subtype PGResult is Address;

   function PQConnectdb(Conninfo: Strings.Chars_Ptr) return PGConn;
   pragma Import (C, PQConnectdb, "PQconnectdb");

   type ConnStatusType is (
                           CONNECTION_OK,
                           CONNECTION_BAD
                           );
   for ConnStatusType use (
                           CONNECTION_OK => 0,
                           CONNECTION_BAD => 1
                           );

   function PQstatus (Conn: PGConn) return ConnStatusType;
   pragma Import (C, PQstatus, "PQstatus");

   procedure PQFinish (Conn: PGConn);
   pragma Import (C, PQFinish, "PQfinish");

   function PQExec (Conn: PGConn; Query: Strings.Chars_Ptr) return PGResult;
   pragma Import (C, PQExec, "PQexec");

   type ExecStatusType is (
     PGRES_EMPTY_QUERY,        -- libpq-fe.h: 68
     PGRES_COMMAND_OK,         -- libpq-fe.h: 69
     PGRES_TUPLES_OK,          -- libpq-fe.h: 72
     PGRES_COPY_OUT,           -- libpq-fe.h: 75
     PGRES_COPY_IN,            -- libpq-fe.h: 76
     PGRES_BAD_RESPONSE,       -- libpq-fe.h: 77
     PGRES_NONFATAL_ERROR,     -- libpq-fe.h: 79
     PGRES_FATAL_ERROR         -- libpq-fe.h: 80
     );
   for ExecStatusType use (
     PGRES_EMPTY_QUERY => 0,        -- libpq-fe.h: 68
     PGRES_COMMAND_OK => 1,         -- libpq-fe.h: 69
     PGRES_TUPLES_OK => 2,          -- libpq-fe.h: 72
     PGRES_COPY_OUT => 3,           -- libpq-fe.h: 75
     PGRES_COPY_IN => 4,            -- libpq-fe.h: 76
     PGRES_BAD_RESPONSE => 5,       -- libpq-fe.h: 77
     PGRES_NONFATAL_ERROR => 6,     -- libpq-fe.h: 79
     PGRES_FATAL_ERROR => 7         -- libpq-fe.h: 80
     );

   function PQResultStatus (Result: PGResult) return ExecStatusType;
   pragma Import (C, PQResultStatus, "PQresultStatus");

   function PQntuples (Result: PGResult) return C.Int;
   pragma Import (C, PQntuples, "PQntuples");

   function PQnfields (Result: PGResult) return C.Int;
   pragma Import (C, PQnfields, "PQnfields");

   function PQfname (Result: PGResult; Field_Index: C.int) return Strings.Chars_Ptr;
   pragma Import (C, PQfname, "PQfname");

   function PQfsize (Result: PGResult; Field_Index: C.int) return C.Int;
   pragma Import (C, PQfsize, "PQfsize");

   function PQgetvalue (Result: PGResult;
                        Tup_Num : C.Int;
                        Field_Num: C.Int) return Strings.Chars_Ptr;
   pragma Import (C, PQgetvalue, "PQgetvalue");

   function PQgetlength (Result: PGResult;
                         Tup_Num : C.Int;
                         Field_Num: C.Int) return C.Int;
   pragma Import (C, PQgetlength, "PQgetlength");

   function PQcmdTuples (Result: PGResult) return Strings.Chars_Ptr;
   pragma Import (C, PQcmdTuples, "PQcmdTuples");

   type PgSQL_Field is record
      null;
   end record;

   --------------
   -- Num_Rows --
   --------------
   function NumRows(Result: PGResult) return Integer is
   begin
      return Integer(PQntuples(Result));
   end NumRows;

   ---------------
   -- NumFields --
   ---------------
   function NumFields(Result: PGResult) return Integer is
   begin
      return Integer(PQnfields(Result));
   end NumFields;

   -----------------
   -- FieldLength --
   -----------------
   function FieldLength(Result: PGresult; Row, Field: Integer) return Natural is
   begin
      return Natural(Pqgetlength(Result,
                                 C.Int(Row),
                                 C.Int(Field)));
   end FieldLength;

   ----------------
   -- FieldValue --
   ----------------
   function FieldValue(Result: PGresult; Row, Field: Integer) return String is
   begin
      return Value(Pqgetvalue(Result,
                              C.Int(Row),
                              C.Int(Field)));
   end Fieldvalue;

   -----------------
   -- Object_Data --
   -----------------
   type Object_Data is record
      Self               : Driver.Handle;
      Connection_Handle : PGConn := Null_Address;
   end record;

   --- ****************************************************************** ---
   --- *****       S T A T E M E N T     H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   ------------------
   -- My_Statement --
   ------------------
   type My_Statement is new Statement.Object with record
      null;
   end record;

   type My_Stmt_Access is access all My_Statement;

   ----------------------
   -- Create_Statement --
   ----------------------
   function Create_Statement(
      This   : in Object;
      Con    : in Connection_ID ) return Statement_ID is
      -- this does more or less nothing, because the interface
      -- of mysql does not requiere any explicit prepare actions
      Result : My_Stmt_Access := new My_Statement( Con );
   begin
      return Statement.Allocate( Statement.Handle(Result) );
   end Create_Statement;

   ----------------------
   -- Delete_Statement --
   ----------------------
   procedure Delete_Statement(
      This : in Object;
      Stmt : in Statement_ID ) is
   begin
      Pragma Debug(Put_Line("DRIVER: delete statement " & Statement_ID'Image(Stmt) ));
      Statement.Deallocate( Stmt );
   end Delete_Statement;

   --- ****************************************************************** ---
   --- ****               R O W    H A N D L I N G                   **** ---
   --- ****************************************************************** ---

   type Field_List is array (Natural range <>) of String_access;
   type Field_List_Access is access all Field_List;

   type My_Row is new Row.Object with record
      RowHandle : Field_List_access;
      Number : Natural := 0;
      Set        : Resultset_ID;
   end record;

   ---------
   -- Get --
   ---------
   procedure Get(
      Row    : in My_Row;
      Pos    : in Positive;
      Result : out String;
      Length : out Natural );

   type My_Row_Access is access My_Row;

   --- ****************************************************************** ---
   --- *****       R E S U L T S E T     H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   type Field_Info_Array is
      array( Positive range 1..Max_Number_Of_Attributes ) of PgSQL_Field;

   --------------------
   -- My_Resultset --
   --------------------
   type My_Resultset is new Resultset.Object with record
      ResultHandle : PGResult;
      Actual_Row : Natural := 0;

      Affected_Rows : Natural          := 0;

      -- here we store information about the attributes of a resultset
      -- which is not needed in the result set class.
      Next_Field    : Positive         := Field_Info_Array'First;
      Fieldinfo     : Field_Info_Array;
   end record;

   type My_Resultset_Access is access all My_Resultset;

   Result_Table : array( Resultset_ID ) of My_Resultset_Access;



   -----------
   -- Fetch --
   -----------
   procedure Fetch(
      This   : in out My_Resultset;
      Result : out Row.Handle;
      Mode   : in Fetch_Mode_Type := Next ) is
      Row    : My_Row_Access;
   begin
       Row := new My_Row( This.Next_Field  );
       Row.Set := ID(This);

       Row.Rowhandle := new Field_List(1..NumFields(This.Resulthandle));

       case Mode is
          when Next     =>
             if This.Actual_Row >= Number_Of_Records( ID (This) ) then
                This.Actual_Row := Number_Of_Records( ID (This) );
                raise End_Of_Resultset;
             else
                This.Actual_Row := This.Actual_Row + 1;
             end if;

          when First    => This.Actual_Row := 0;

          when Last     => This.Actual_Row := Number_Of_Records(ID(This)) -1;

          when Previous =>
             if This.Actual_Row > 0 then
                This.Actual_Row := This.Actual_Row - 1;            -- throw begin_of_resultset?
             end if;
       end case;

       Row.Number := This.Actual_Row;
       Result := GNU.DB.ADBC.Row.Handle(Row);
   end Fetch;

   ----------------------
   -- Create_Resultset --
   ----------------------
   function Create_Resultset(
      This   : in Object;
      Stmt   : in Statement_ID ) return Resultset_ID is
      Result : My_Resultset_Access := new My_Resultset(Stmt) ;
      Id     : Resultset_ID;
   begin
      Result.Resulthandle := Null_Address;

      Id := Resultset.Allocate( Resultset.Handle(Result) );
      Result_Table(Id) := Result;

      return Id;
   end Create_Resultset;

   ----------------------
   -- Delete_Resultset --
   ----------------------
   procedure Delete_Resultset(
      This   : in Object;
      Result : in Resultset_ID ) is
   begin
      Pragma Debug(Put_Line("DRIVER: delete resultset " & Resultset_ID'Image(Result) ));
      null;
   end Delete_Resultset;

   ---------
   -- Get --
   ---------
   procedure Get(
      Row    : in My_Row;
      Pos    : in Positive;
      Result : out String;
      Length : out Natural ) is
      -- get a field from the resultset
      Set : My_Resultset renames
               My_Resultset_Access( Resultset.Get_Handle( Row.Set ) ).all;
      -- index fixing - positions in PostgreSQL start at 0
      DPos : constant Integer := -1;
      DRow : constant Integer := 0;
   begin
      Length := Natural(FieldLength(Set.ResultHandle,
                                    Row.Number + DRow,
                                    Pos + DPos));

      if Length /= 0 then
         declare
            Str : String(1..Length);
         begin
            Str := FieldValue(Set.ResultHandle,
                              Row.Number + DRow,
                              Pos + DPos);
            Result(1..Length) := Str(1..Length);
         end;
      end if;
  end Get;

   --- ****************************************************************** ---
   --- *****       C O N N E C T I O N   H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   -------------
   -- Connect --
   -------------
   procedure Connect(
                     This     : in out Object;
                     User     : in String;
                     Password : in String;
                     Database : in String) is
      -- connecting to an actual database server
      Data     : Object_Data_Access renames This.Data;
      Conn_Str : Unbounded_String;

      RC       : ConnStatusType;
   begin
      if Data = null then
         Data := new Object_Data;
      end if;

      Conn_Str := To_Unbounded_String("host="&"localhost");
      Conn_Str := Conn_Str & " user="&User;
      if Database /= Default_Database then
         Conn_Str := Conn_Str & " dbname=" & Database;
      end if;
      if Password /= Default_Password then
         Conn_Str := Conn_Str & " password=" & Password;
      end if;

      Data.Connection_Handle := PQConnectdb (New_String(To_String(Conn_str)));

      RC := PQstatus(Data.Connection_Handle);
      if Data.Connection_Handle = Null_Address or RC = CONNECTION_BAD then
         raise Connection_Failure;
      end if;

      Pragma Debug(Put_Line("DRIVER:[PgSQL] connect to database complete"));
   end Connect;

   ----------------
   -- Disconnect --
   ----------------
   procedure Disconnect(
      This : in out Object ) is
      -- diconnect from the data base
      Data : Object_Data_Access renames This.Data;
   begin
      Pqfinish (Data.Connection_Handle);
      Pragma Debug(Put_Line("DRIVER: disconnect"));
   end Disconnect;

   --- ****************************************************************** ---
   --- *****       C O N N E C T I O N   H A N D L I N G           ****** ---
   --- ****************************************************************** ---

   -------------
   -- Prepare --
   -------------
   procedure Prepare(
      This : in out Object;
      Stmt : in Statement_ID ) is
      -- there is no functionaltiy for MySQL required since it does
      -- not support the preperation of statement.
      St   : My_Statement renames My_Stmt_Access( Get_Handle(Stmt) ).all;
   begin
      Pragma Debug(Put_Line("DRIVER: prepare does nothing"));
      null;
   end Prepare;

   ------------
   -- Expand --
   ------------
   function Expand(
      This : in Object;
      -- This procedure returns the SQL ASCII representation of the host
      -- variable for an SQL statement.
      -- In case of ODBC this procedure returns allways a '?' character
      -- and the Bind method below does the trick of connecting variables
      -- with the data base.
      -- In all other cases the ascii string will be returned.
      V    : in Hostvariable.Handle ) return String is
   begin
      if V.all'Tag = SQL_String'Tag then
         return "'" & Value( SQL_String(V.all) ) & "'";
      elsif V.all'Tag = SQL_Integer'Tag then
         return Integer'Image( Value(SQL_Integer(V.all)) );
      else
         return "?";
      end if;
   end Expand;

   ----------
   -- Bind --
   ----------
   procedure Bind_Host_Variable(
      This : in Object;
      Stmt : in Statement_ID;
      V    : in Hostvariable.Handle) is
      -- This procedure will be called before the statement is executed. The
      -- main application is the allocation of defered buffers as for example
      -- used by the ODBC interface.
      -- The mapping between Ado and DBCS specific representation is done at
      -- this point.

      procedure Bind_SQL_String(
         S : in String ) is
      begin
         Pragma Debug(Put_Line("DRIVER: Bind SQL_String does nothing, value='" & S & "'" ));
         null;
      end Bind_SQL_String;

      procedure Bind_SQL_Integer(
         S : in Integer ) is
      begin
         Pragma Debug(Put_Line("DRIVER: Bind SQL_Integer does nothing, value =" & Integer'Image(S)));
         null;
      end Bind_SQL_Integer;

   begin
      if V.all'Tag = SQL_String'Tag then
         Bind_SQL_String( Value( SQL_String(V.all) ) );
      elsif V.all'Tag = SQL_Integer'Tag then
         Bind_SQL_Integer( Value( SQL_Integer(V.all) ) );
      end if;
   end Bind_Host_Variable;

   --------------------
   -- Get_Host_Value --
   --------------------
   procedure Get_Host_Value(
      This : in Object;
      Stmt : in Statement_ID;
      V    : in Hostvariable.Handle) is
   begin
      Pragma Debug(Put_Line("DRIVER:Get_Value does nothing"));
      null;
   end Get_Host_Value;

   function Error_Msg (Result: PGResult) return Chars_Ptr;
   pragma Import (C, Error_Msg, "PQresultErrorMessage");

   -------------
   -- Execute --
   -------------
   procedure Execute_Statement(
      This       : in Object;
      Result     : in Resultset_ID;
      Stmt       : in Statement_ID ) is
      -- execute a statement:
      --   S.1 - Resolve unbounde variables
      --   S.2 - Execute the actuial query
      --   S.3 - Fetch the field information
      --   S.4 - Fetch the number of affected rows if the query has
      --         not created any result set.
      Data       : Object_Data_Access renames This.Data;
      Set        : My_Resultset renames
                      My_Resultset_Access( Resultset.Get_Handle( Result ) ).all;

      Q          : constant String := Query(Stmt);
      Res        : PGResult;
      RC         : ExecStatusType;
      Field      : PgSQL_Field;
   begin
      Res := PQExec( Data.Connection_Handle, New_String(Q));

      RC := PQResultStatus(Res);
      if RC = PGRES_EMPTY_QUERY then
         raise Empty_Resultset;
      end if;

      if RC = PGRES_FATAL_ERROR or RC = PGRES_NONFATAL_ERROR then
         Pragma Debug(Put_Line(Value(Error_Msg(Res))));
         raise Empty_Resultset;                           -- should return an error exception
      end if;

      -- fetch the result set                                       *** S.3 ***
      if RC = PGRES_TUPLES_OK then
         if NumRows( RES ) < 1 then
            raise Empty_Resultset;
         end if;
         Number_Of_Records( Result, NumRows(RES));
         Set.ResultHandle := Res;

         for I in 0..PQnfields(Set.ResultHandle)-1 loop
            Attribute( Result, Value( PQfname( Set.ResultHandle, I)));

            Set.Fieldinfo( Set.Next_Field ) := Field;
            Set.Next_Field := Set.Next_Field + 1;

            exit when not ( Set.Next_Field in Set.Fieldinfo'Range );
         end loop;
      elsif RC = PGRES_COMMAND_OK then                  -- *** S.4 ***
         Set.Affected_Rows := Natural'Value(Value (PQcmdTuples( Set.ResultHandle )));
      else
         null;
         -- return error
      end if;
   end Execute_Statement;

   ------------
   -- Create --
   ------------
   type PgSQL_Access is access all Object;

   function Create return Driver.Handle is

      Db   : PgSQL_Access := new Object;
      Data : Object_Data_Access := new Object_Data;
   begin
      Data.Self := Driver.Handle(Db);
      Db.Data := Data;
      return Driver.Handle(Db);
   end Create;

end GNU.DB.ADBC.Driver.PgSQL;

