	; 18f14k22 gpsim regression test
	;
	; The purpose of this test is to verify that the 16bit processors
	; is implemented correctly. Here's a list of specific things
	; tested:	
	;
	; 1) Reset conditions
	; 2) WDT
	; 3) Sleep
	; 4) Wakeup on PIN change.

	list	p=18f14k22
        include <p18f14k22.inc>
        include <coff.inc>

  CONFIG  FOSC = IRC,  WDTPS = 32, WDTEN = ON

        radix   dec

; Printf Command
.command macro x
  .direct "C", x
  endm

GPR_DATA  UDATA
ResetSequence RES 1
optionShadow  RES 1
w_temp RES 1
status_temp RES 1
temp1 RES 1
intx RES 1

  GLOBAL optionShadow, ResetSequence
  GLOBAL temp1, intx


    ; Define the reset conditions to be checked.

eRSTSequence_PowerOnReset	equ	1
eRSTSequence_AwakeMCLR		equ	2
eRSTSequence_AwakeWDT		equ	3
eRSTSequence_AwakeIO		equ	4
eRSTSequence_WDTTimeOut		equ	5
eRSTSequence_STKUNF		equ	6
eRSTSequence_STKFUL		equ	7

;----------------------------------------------------------------------
;   ********************* STARTUP LOCATION  ***************************
;----------------------------------------------------------------------
START  CODE    0x000                    ; 


;############################################################
;# Create a stimulus to simulate a switch
;

   .sim "p18f14k22.xpos = 84"
   .sim "p18f14k22.ypos = 96"

   .sim "module lib libgpsim_modules"
   .sim "module load pulsegen P_INTF"
   .sim "module load pulsegen P_INTF2"
   .sim "module load pulsegen P_RBIF"
   .sim "module load pulsegen RESET"

   .sim "RESET.initial = 5.0"
   .sim "RESET.xpos = 84"
   .sim "RESET.ypos = 36"

   .sim "P_INTF.clear = 0"   ; At cycle 0, 
   .sim "P_INTF.xpos = 240"
   .sim "P_INTF.ypos = 72"

   .sim "P_INTF2.clear = 0"   ; At cycle 0, 
   .sim "P_INTF2.xpos = 240"
   .sim "P_INTF2.ypos = 132"

   .sim "P_RBIF.clear = 0"   ; At cycle 0, 
   .sim "P_RBIF.set = 1"
   .sim "P_RBIF.xpos = 240"
   .sim "P_RBIF.ypos = 192"

  ;############################################################

  .sim "node nINTF"
  .sim "attach nINTF porta0 P_INTF.pin"

  .sim "node nINTF2"
  .sim "attach nINTF2 porta2 P_INTF2.pin"

  .sim "node nRBIF"
  .sim "attach nRBIF portb4 P_RBIF.pin"

  .sim "node nRESET"
;  .sim "attach nRESET p16f84.MCLR RESET.pin"
  .sim "attach nRESET MCLR RESET.pin"

  ;############################################################
  .sim "symbol resetCounter=1"

   ;  NOT_GPWU = 0  - Enable wakeup if I/O pins 0,1, or 3 change states
   ;  NOT_GPPU = 1  - Disable weak pullups on I/O pins 0,1, and 3 
   ;  TOCS = 0 - Let the clock source for TMR0 be the internal fosc/4
   ;  TOSE = 0 - don't care - TMR0 source edge.
   ;  PSA = 1 - assign Prescale to the WDT
   ;  PS2:0 = 000 - prescale = 2^0=1

  .sim "optionShadow=8"      ; PSA=1

  .sim "ResetSequence=0"

  .sim ".BreakOnReset = false"
  .sim "p18f14k22.frequency=100000"

  ; Set a cycle break point far in the future in case the resets fail.
  .sim "break c 0x1000000"



RESET_VECTOR  CODE    0x000              ; processor reset vector

        movlw  high  start               ; load upper byte of 'start' label
        movwf  PCLATH                    ; initialize PCLATH
        goto   start                     ; go to beginning of program




;------------------------------------------------------------------------
;
;  Interrupt Vector
;
;------------------------------------------------------------------------

INT_VECTOR   CODE    0x008               ; high interrupt vector location
        movwf   w_temp
        swapf   STATUS,W
        movwf   status_temp
	goto	check_rbi

LOW_INT_VECTOR CODE 0x018               ; Low priority interrupt vector
        movwf   w_temp
        swapf   STATUS,W
        movwf   status_temp

        btfsc   INTCON3,INT2IF
         btfss  INTCON3,INT2IE
	   goto bad_low

	bsf	intx,2
	bcf	INTCON3,INT2IF
	goto	exit_int

bad_low:
  .assert  "\"*** FAILED 18f14k22 unexpected low interrupt\""
	nop
        goto    $


check_rbi:
        btfsc   INTCON,RBIF
         btfss  INTCON,RBIE
          goto  check_int0

;        bsf     temp5,1         ;Set a flag to indicate rb4-7 int occured
        bcf     INTCON,RBIF
	movf	PORTB,w
        
check_int0:
        btfsc   INTCON,INT0IF
         btfss  INTCON,INT0IE
          goto  check_t0

	bsf	intx,0		; Set flag int0 interrupt
        bcf     INTCON,INT0IF
	goto	exit_int

check_t0:
        btfsc   INTCON,T0IF
         btfss  INTCON,T0IE
	   goto	unknown

    ;; tmr0 has rolled over
        
        bcf     INTCON,T0IF     ; Clear the pending interrupt
        bsf     temp1,0         ; Set a flag to indicate rollover
        goto    exit_int
                
unknown:
    .assert "\"Unknown interrupt\""
	nop
exit_int:               

        swapf   status_temp,w
        movwf   STATUS
        swapf   w_temp,f
        swapf   w_temp,w
        retfie





bSWITCH equ 0

;----------------------------------------------------------------------
;   ******************* MAIN CODE START LOCATION  ******************
;----------------------------------------------------------------------
MAIN    CODE
start

  .assert "trisa==0x3f, \"*** FAILED 18f14k22 reset bad TRISA\""
	nop
  .assert "trisb==0xf0, \"*** FAILED 18f14k22 reset bad TRISB\""
	nop
  .assert "trisc==0xff, \"*** FAILED 18f14k22 reset bad TRISC\""
	nop
   ; OPTION register setup

        MOVF    optionShadow,W  ;optionShadow is initialized by the gpsim script

   ; GPIO setup
   ; If this is a PowerOn reset, then the state of the GPIO output register
   ; is unknown. If this is not a PowerOn reset then the GPIO pins have
   ; the same state they had prior to the reset.

   ;
   ; TRIS register setup - The I/O pins are always configured as Inputs after
   ; any reset.


        MOVLW   (1<<RB4)     ;RB4 are the only inputs.
        MOVWF   TRISB     ;
	MOVLW	(1<<RA0)| (1<<RA2)	; RA0,RA2 input for INT0, INT2
        MOVWF    TRISA 
        CLRF    STATUS
	CLRF	TRISC
	MOVLW	(1<<TMR0ON) | (1<<T08BIT)
	MOVWF	T0CON	; start T0
  	clrf	ANSEL
	clrf	IOCB
  	clrf	ANSELH
	movlw	0xf0
	movwf	IOCB

   ; Now examine why the RESET condition occurred.

	BTFSS	RCON,NOT_TO
	 goto   TO_is_low

	BTFSS	RCON,NOT_PD
	 goto   AwakeMCLR	;/TO=1, /PD=0 ==> /MCLR while sleeping
                                ;            -or- Interrupt wake-up from sleep

        btfsc   STKPTR,STKUNF   ; Stack undeflow has occured
	goto	stack_underflow

        btfsc   STKPTR,STKFUL   ; Stack overflow has occured
	goto	stack_overflow

	goto	PowerOnReset	;/TO=1, /PD=1 ==> Power has just been applied

TO_is_low

	BTFSS	RCON,NOT_PD
	 goto   AwakeWDT	;/TO=0, /PD=0 ==> WDT while sleeping

	goto	WDTTimeOut	;/TO=0, /PD=1 ==> WDT while not sleeping


	BTFSC	INTCON, RBIF	;Did we just wake up from SLEEP due
         goto   AwakeIO		;to a change in the I/O pins?


;========================================================================
;
;  PowerOnReset
;
;========================================================================
PowerOnReset:

        movf    ResetSequence,W
        XORLW   eRSTSequence_AwakeMCLR
        skpnz
        goto   to_underflow

        movf    PORTB,W         ;Clear RBIF
  .assert "(intcon&1)==0, \"*** FAILED to clear INTCON\""
        nop

  .assert "resetCounter==1,\"*** FAILED Power On Reset\""
	nop
	MOVLW	eRSTSequence_PowerOnReset
	MOVWF	ResetSequence
	CLRWDT
  .command "resetCounter = resetCounter+1"
	nop
	SLEEP

   ; The WDT should cause a reset. So we shouldn't fall through
   ; (note, the instruction after the SLEEP should not be executed).
   ; RRR WDT will not cause a reset but will fall through
        
        nop
  .assert "trisa==0x0d, \"*** FAILED 18f14k22 WDT sleep bad TRISA\""
	nop
  .assert "trisb==0x10, \"*** FAILED 18f14k22 WDT sleep bad TRISB\""
	nop
  .assert "trisc==0x00, \"*** FAILED 18f14k22 WDT sleep bad TRISC\""
	nop
;RRR  .command "resetCounter = resetCounter+1"
        nop
	nop
;========================================================================
;
;  AwakeWDT - WDT timed out while we were sleeping.
;
;========================================================================
AwakeWDT:

  .assert "resetCounter==2,\"*** FAILED WDT Reset\""
	nop
	MOVLW	eRSTSequence_AwakeWDT
	MOVWF	ResetSequence

  .command "resetCounter = resetCounter+1"
	nop
	CLRWDT

    ; loop until WDT times out
here:   goto    here


;========================================================================
;
;  WDTTimeOut - WatchDog timer timed out while we were awake.
;
;========================================================================
WDTTimeOut:

  .assert "resetCounter==3,\"*** FAILED WDT timeout\""
	nop
	MOVLW	eRSTSequence_WDTTimeOut
	MOVWF	ResetSequence

	CLRWDT

  .command "P_RBIF.clear = cycles+100"
        nop

    ; Keep interrupts enabled, but turn on RBIE. Changes on
    ; RB4-RB7 should wake us from sleep 
        MOVLW   1<<RBIE
        MOVWF   INTCON

	MOVF	TMR0L,W
	SLEEP

        nop    ; the processor should idle at this instruction.
  .assert "(tmr0l - W) <= 2, \"*** FAILED 18f14k22 reset - TMR0 stops during sleep\""
        nop
  .assert "(intcon & 1) == 1, \"*** FAILED 18f14k22 Wake from sleep not RBIF\""
	nop
  .command "resetCounter = resetCounter+1"
	nop

;=======================================================================
;
;	INT0
;
;=======================================================================
	CLRF	IOCA
	clrf	INTCON
  .command "P_INTF.set = cycles+4"
        nop
	sleep		; INT0 should not wake sleep, WDT instead
	nop
  .assert "(rcon & 8) == 0, \"*** FAILED p18f14k22 INT0 INT0IE=0 WDT wake from sleep\""
	nop

	; test wake from sleep, no interrupt
        MOVLW   (1<<INT0IE)
        MOVWF   INTCON
	bcf	INTCON2,INTEDG0	; INT0 on falling edge
  .command "P_INTF.clear= cycles+4"
	nop
	sleep
	nop
  .assert "(((rcon & 8) == 8) && (intx == 0)), \"*** FAILED p18f14k22 INT0 INT0IE=1 GIE=0\""
	nop
	; test int0 inturrupt
        MOVLW   (1<<INT0IE | 1<<GIE)
        MOVWF   INTCON
	bsf	INTCON2,INTEDG0	; INT0 on rising edge
  .command "P_INTF.set= cycles+4"
	nop
	sleep
	nop
  .assert "(((rcon & 8) == 8) && (intx == 1)), \"*** FAILED p18f14k22 INT0 INT0IE=1 GIE=1\""
	nop

;=======================================================================
;
;	INT2
;
;=======================================================================
	clrf	INTCON
	bcf	INTCON3,INT2IE
  .command "P_INTF2.set = cycles+4"
        nop
	sleep		; INT2 should not wake sleep, WDT instead
	nop
  .assert "(rcon & 8) == 0, \"*** FAILED p18f14k22 INT2 INT2IE=0 WDT wake from sleep\""
	nop

	; test wake from sleep, no interrupt
	clrf	intx
	movlw	1<<INT2IE	; this also sets low priority interrupts
	movwf	INTCON3
	bcf	INTCON2,INTEDG2	; INT2 on falling edge
  .command "P_INTF2.clear= cycles+4"
	nop
	sleep
	nop
  .assert "(((rcon & 8) == 8) && (intx == 0)), \"*** FAILED p18f14k22 INT2 INT2IE=1 GIE=0\""
	nop
	; test int2 interrupt
	clrf	intx
	bcf	INTCON3,INT2IF
        MOVLW   (1<<GIE)
        MOVWF   INTCON
	bsf	INTCON3,INT2IE
	bsf	INTCON2,INTEDG2	; INT2 on rising edge
  .command "P_INTF2.set= cycles+4"
	nop
	sleep
	nop
  .assert "(((rcon & 8) == 8) && (intx == 4)), \"*** FAILED p18f14k22 INT2 INT2IE=1 GIE=1\""
	nop

;========================================================================
;
;  AwakeIO - an I/O pin changed states to awaken us from sleep.
;
;========================================================================
AwakeIO:
  .assert "resetCounter==4,\"*** FAILED Wakeup on I/O pin change\""
	nop
	MOVLW	eRSTSequence_AwakeIO
	MOVWF	ResetSequence

  .command "resetCounter = resetCounter+1"
	nop

	CLRWDT

    ; reset the processor by driving /MCLR low for a few cycles 
  .command "RESET.clear = cycles+100"
  .command "RESET.set = cycles+110"
        nop

	SLEEP

;========================================================================
;
;  AwakeMCLR - MCLR went low while we were sleeping.
;
;========================================================================
AwakeMCLR:
  .assert "resetCounter==5,\"*** FAILED /MCLR Reset\""
	nop

	MOVLW	eRSTSequence_AwakeMCLR
	MOVWF	ResetSequence

  .command "resetCounter = resetCounter+1"
	nop
	CLRWDT

    ; reset the processor by driving /MCLR low for a few cycles 
  .command "RESET.clear = cycles+100"
  .command "RESET.set = cycles+110"
        nop

waitForMCLR:    goto waitForMCLR







stack_underflow:
  .assert "resetCounter==6,\"*** FAILED Stack undeflow Reset\""
	nop
  .command "resetCounter = resetCounter+1"
	nop
	CLRWDT
	bcf     STKPTR,STKUNF
	clrf	temp1
	call	full

full:
	incf	temp1,F
	call full
	return		; never gets executed
	nop

stack_overflow:
  .assert "resetCounter==7,\"*** FAILED Stack overflow Reset\""
	nop
  .assert "temp1==31, \"*** FAILED Stack overflow size\""
	nop
  .assert  "\"*** PASSED 18f14k22 Sleep and Reset test\""
	nop
	nop

to_underflow:
	return	; RRR stack undeflow
	
	end
