/*
 * Copyright (C) 2025 UBports Foundation
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License 3 as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program. If not, see http://www.gnu.org/licenses/.
 */

import QtQuick 2.12
import Lomiri.Components 1.3
import QtQuick.Layouts 1.12
import Lomiri.Components.Popups 1.3

import "../components" as Components
import "../actions" as FMActions
import "../views" as Views
import "../backend"

Item {
    id: rootItem

    property var folderPage
    property FolderListModel folderModel
    property var selectionManager: folderModel.model.selectionObject
    property var openDefault
    property var fileOperationDialog
    property bool selectionMode: false
    property bool contentHubMode: false
    property bool importMode: false
    property bool narrowAspect: false
    property bool searchBarIsShown: false

    readonly property var defaultLeadingActions: folderPage.showPanelAction.visible ? folderPage.showPanelAction : placesBookmarkAction
    readonly property var defaultTrailingActions: {
        if (narrowAspect) {
            return [
                searchAction, newAction
                , addBookmarkAction, propertiesAction
                , openInTerminalAction
            ]
        } else {
            return [
                openInTerminalAction, propertiesAction
                , addBookmarkAction, newAction, searchAction
            ]
        }
    }

    readonly property var defaultHeaderSecondaryTrailingActions: [ clearClipboardAction, pasteAction ]

    readonly property var extensionLeadingActions: [ goForwardAction, goBackAction ]

    readonly property var extensionTrailingActions: narrowAspect ? [ settingsAction ]
    : [
        settingsAction, viewTypeAction
    ]

    readonly property var selectionLeadingActions: [
        cancelSelectionModeAction
    ]

    readonly property var selectionTrailingActions: {
            if (narrowAspect) {
                return [
                    cutAction, copyAction, selectUnselectAllAction
                    , openAdvancedAction, renameAction, deleteAction
                    , propertiesAction
                ]
            } else {
                return [
                    selectUnselectAllAction, copyAction
                    , cutAction, openAdvancedAction, renameAction
                    , deleteAction, propertiesAction
                ]
            }
    }

    readonly property var pickerModeTrailingActions: [
        cancelPickerModeAction, selectAction, newAction, searchAction
    ]

    property alias goBackAction: goBackAction
    property alias goForwardAction: goForwardAction
    property alias placesBookmarkAction: placesBookmarkAction
    property alias searchAction: searchAction
    property alias copyAction: copyAction

    property var __delegateActions: Views.FolderDelegateActions {
        folderListPage: rootItem.folderModel
        folderModel: rootItem.folderPage.folderModel
        fileOperationDialog: rootItem.fileOperationDialog
        openDefault: rootItem.openDefault
    }

    signal enterSearch

    function enterSelectionMode() {
        fileSelectorMode = true
        fileSelector.fileSelectorComponent = pageStack
    }

    ActionContext {
        id: globalContext

        active: true
        actions: [
            FMActions.GoBack {
                id: goBackAction
                shortcut: "Alt+Left"
                enabled: folderModel.canGoBack
                onTriggered: folderModel.goBack()
            },
            FMActions.GoForward {
                id: goForwardAction
                shortcut: "Alt+Right"
                enabled: folderModel.canGoForward
                onTriggered: folderModel.goForward()
            },
            FMActions.GoUp {
                id: goUpAction
                visible: folderModel.canGoUp
                onTriggered: folderModel.goUp()
            },
            FMActions.GoHome {
                id: goHomeAction
                visible: folderModel.canGoHome
                onTriggered: folderModel.goHome()
            },
            FMActions.ViewType {
                id: viewTypeAction
                isGrid: globalSettings.viewMethod == 1
                onTriggered: {
                    if (isGrid) {
                        globalSettings.viewMethod = 0
                    } else {
                        globalSettings.viewMethod = 1
                    }
                }
            },
            FMActions.Settings {
                id: settingsAction
                onTriggered: PopupUtils.open(Qt.resolvedUrl("ViewPopover.qml"), mainView, { folderListModel: folderModel.model })
            },
            FMActions.Properties {
                id: propertiesAction
                visible: !rootItem.searchBarIsShown
                shortcut: "Alt+Return"

                onTriggered: {
                    if (selectionManager.counter == 1) {
                        var props = {
                            "model": folderModel.primSelItem
                        }
                        PopupUtils.open(Qt.resolvedUrl("FileDetailsPopover.qml"), mainView, props)
                    } else {
                        print(text)
                        PopupUtils.open(Qt.resolvedUrl("FileDetailsPopover.qml"), mainView,{ "model": folderModel.model })
                    }
                }
            },
            FMActions.Search {
                id: searchAction
                showSearchBar: rootItem.searchBarIsShown
                shortcut: rootItem.searchBarIsShown ? StandardKey.Cancel: StandardKey.Find
                enabled: rootItem.contentHubMode || !rootItem.selectionMode
                onTriggered: rootItem.enterSearch()
            },
            FMActions.NewItem {
                id: newAction
                // Hide when in export mode since we don't expect users to create new folders in this case
                visible: !rootItem.searchBarIsShown && !(rootItem.contentHubMode && !rootItem.importMode)
                property bool smallText: true
                enabled: folderModel.model.isWritable
                shortcut: "Ctrl+Shift+N"
                onTriggered: {
                    print(text)
                    let props = { folderPage: folderListPage, folderModel: folderModel.model, folderOnly: rootItem.contentHubMode && rootItem.importMode }
                    PopupUtils.open(Qt.resolvedUrl("../dialogs/CreateItemDialog.qml"), mainView, props)
                }
            },
            FMActions.FileClearSelection {
                id: clearClipboardAction
                clipboardUrlsCounter: folderModel.model.clipboardUrlsCounter
                visible: folderModel.model.clipboardUrlsCounter > 0 && !rootItem.searchBarIsShown
                onTriggered: {
                    console.log("Clearing clipboard")
                    folderModel.model.clearClipboard()
                    folderPage.tooltipMsg = i18n.tr("Cleared clipboard")
                }
            },
            FMActions.FilePaste {
                id: pasteAction
                property bool smallText: true

                clipboardUrlsCounter: folderModel.model.clipboardUrlsCounter
                visible: folderModel.model.clipboardUrlsCounter > 0 && !rootItem.searchBarIsShown
                shortcut: StandardKey.Paste
                onTriggered: {
                    console.log("Pasting to current folder items of count " + folderModel.model.clipboardUrlsCounter)
                    fileOperationDialog.startOperation(i18n.tr("Paste files"))
                    folderModel.model.paste()
                    folderPage.tooltipMsg = i18n.tr("Pasted item", "Pasted items", folderModel.model.clipboardUrlsCounter)

                    // We want this in a mobile environment.
                    folderModel.model.clearClipboard()
                }
            },
            FMActions.AddBookmark {
                id: addBookmarkAction
                visible: !folderModel.model.clipboardUrlsCounter > 0 && !rootItem.searchBarIsShown
                shortcut: "Ctrl+D"
                onTriggered: {
                    print(text)
                    folderModel.places.addLocation(folderModel.model.path)
                    folderPage.tooltipMsg = i18n.tr("Added '%1' to Places").arg(folderModel.model.fileName)

                }
            },
            FMActions.Terminal {
                id: openInTerminalAction
                visible: !rootItem.searchBarIsShown
                onTriggered: {
                    print(text)
                    Qt.openUrlExternally("terminal://?path=" + folderModel.model.path)
                }
            },
            FMActions.PlacesBookmarks {
                id: placesBookmarkAction
                property var pushedPage

                shortcut: "Ctrl+Shift+O"
                enabled: !folderPage.sidebarActive

                onTriggered: {
                    if (pushedPage) {
                        pushedPage.pageStack.pop()
                        pushedPage = null
                    } else {
                        const pp = pageStack.push(Qt.resolvedUrl("PlacesPage.qml"), { folderModel: rootItem.folderModel })
                        pushedPage = pp
                        pp.pathClicked.connect(function() {
                            pp.pageStack.pop()
                            pushedPage = null
                        })
                    }
                }
            },
            FMActions.SelectUnselectAll {
                id: selectUnselectAllAction
                selectedAll: selectionManager.selectedAll
                shortcut: {
                    if (rootItem.selectionMode && selectedAll) {
                        return StandardKey.Deselect
                    }

                    return StandardKey.SelectAll
                        
                }
                onTriggered: {
                    if (selectionManager.selectedAll) {
                        selectionManager.clear()
                    } else {
                        if (!rootItem.selectionMode) {
                            rootItem.enterSelectionMode()
                        }
                        selectionManager.selectAll()
                    }
                }
            }
        ]
    }

    ActionContext {
        id: selectionContext

        property bool actionsEnabled: (rootItem.selectionManager.counter > 0) || (folderSelectorMode && folderModel.model.isWritable)
        property bool actionsVisible: selectionMode

        active: rootItem.selectionMode
        actions: [
            Action {
                id: cancelSelectionModeAction
                text: i18n.tr("Cancel")
                iconName: "close"
                // TODO: This triggers first when closing a dialog with Escape
                shortcut: StandardKey.Cancel
                onTriggered: {
                    console.log("FileSelector cancelled")
                    rootItem.selectionManager.clear()
                    fileSelectorMode = false
                    fileSelector.fileSelectorComponent = null
                }
            },
            FMActions.FileCopy {
                id: copyAction
                property bool smallText: true
                enabled: selectionContext.actionsEnabled
                visible: selectionContext.actionsVisible
                shortcut: StandardKey.Copy
                onTriggered: {
                    var selectedAbsPaths = rootItem.selectionManager.selectedAbsFilePaths();
                    folderModel.model.copyPaths(selectedAbsPaths)
                    rootItem.selectionManager.clear()
                    fileSelectorMode = false
                    fileSelector.fileSelectorComponent = null
                }
            },
            FMActions.FileCut {
                id: cutAction
                property bool smallText: true
                enabled: selectionContext.actionsEnabled
                visible: selectionContext.actionsVisible && folderModel.model.isWritable
                shortcut: StandardKey.Cut
                onTriggered: {
                    var selectedAbsPaths = rootItem.selectionManager.selectedAbsFilePaths();
                    folderModel.model.cutPaths(selectedAbsPaths)
                    rootItem.selectionManager.clear()
                    fileSelectorMode = false
                    fileSelector.fileSelectorComponent = null
                }
            },
            FMActions.Delete {
                id: deleteAction
                property bool smallText: true
                enabled: selectionContext.actionsEnabled
                visible: selectionContext.actionsVisible && folderModel.model.isWritable
                shortcut: StandardKey.Delete
                onTriggered: {
                    var selectedAbsPaths = rootItem.selectionManager.selectedAbsFilePaths();
                    var props = {
                        "paths" : selectedAbsPaths,
                        "selectionManager" : rootItem.selectionManager,
                        "folderModel": folderModel.model,
                        "fileOperationDialog": fileOperationDialogObj
                    }

                    PopupUtils.open(Qt.resolvedUrl("../dialogs/ConfirmMultipleDeleteDialog.qml"), mainView, props)
                }
            },
            FMActions.OpenAdvanced {
                id: openAdvancedAction
                property bool smallText: true
                enabled: selectionContext.actionsEnabled
                visible: selectionContext.actionsVisible && !folderModel.primSelItem.isBrowsable && rootItem.openDefault && rootItem.selectionManager.counter == 1
                onTriggered: __delegateActions.openAdvanced(folderModel.primSelItem)
            },
            FMActions.Rename {
                id: renameAction
                visible: selectionContext.actionsVisible && folderModel.model.isWritable && selectionManager.counter == 1
                shortcut: "F2"
                onTriggered: {
                    var props = {
                        "modelRow" : folderModel.primSelItem.index,
                        "inputText" : folderModel.primSelItem.fileName,
                        "placeholderText" : folderModel.primSelItem.fileName,
                        "folderModel": folderModel.model
                    }

                    var popup = PopupUtils.open(Qt.resolvedUrl("../dialogs/ConfirmRenameDialog.qml"), mainView, props)

                    popup.accepted.connect(function(inputText) {
                        console.log("Rename accepted", inputText)
                        if (inputText !== '') {
                            console.log("Rename commensed, modelRow/inputText", fileOperationDialog.index, inputText.trim())
                            if (folderModel.model.rename(folderModel.primSelItem.index, inputText.trim()) === false) {
                                var props = {
                                    title: i18n.tr("Could not rename"),
                                    text: i18n.tr("Insufficient permissions, name contains special chars (e.g. '/'), or already exists")
                                }
                                PopupUtils.open(Qt.resolvedUrl("../dialogs/NotifyDialog.qml"), mainView, props)
                            }
                            selectionManager.clear()
                            fileSelectorMode = false
                        } else {
                            console.log("Empty new name given, ignored")
                        }
                    })
                }
            }
        ]
    }

    ActionContext {
        id: pickerModeContext

        property bool actionsEnabled: (rootItem.selectionManager.counter > 0) || (folderSelectorMode && folderModel.model.isWritable)
        property bool actionsVisible: selectionMode

        active: rootItem.contentHubMode
        actions: [
            FMActions.Cancel {
                id: cancelPickerModeAction
                text: i18n.tr("Cancel")
                iconName: "close"
                shortcut: StandardKey.Cancel
                onTriggered: {
                    console.log("FileSelector cancelled")
                    mainView.cancelFileSelector()
                    rootItem.selectionManager.clear()
                }
            },
            FMActions.Select {
                id: selectAction
                enabled: rootItem.selectionManager.counter > 0 || rootItem.importMode
                onTriggered: {
                    var selectedAbsUrls = []
                    if (folderSelectorMode) {
                        selectedAbsUrls = [ folderModel.path ]
                    } else {
                        var selectedAbsPaths = rootItem.selectionManager.selectedAbsFilePaths();
                        // For now support only selection in filesystem
                        selectedAbsUrls = selectedAbsPaths.map(function(item) {
                            return "file://" + item;
                        });
                    }
                    console.log("FileSelector OK clicked, selected items: " + selectedAbsUrls)
                    acceptFileSelector(selectedAbsUrls)
                }
            }
        ]
    }
}
