/*
 * JBoss, the OpenSource J2EE webOS
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache.lock;

import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;

/**
 * Transaction isolation level of READ-UNCOMMITTED. Reads always succeed (NullLock), whereas writes are exclusive.
 * It prevents none of the dirty read, non-repeatable read, or phantom read.
 *
 * @author Ben Wang
 * @version $Revision: 3125 $
 */
public class LockStrategyReadUncommitted implements LockStrategy
{
   private SemaphoreLock wLock_;
   private Lock rLock_; // Null lock will always return true

   public LockStrategyReadUncommitted()
   {
      wLock_ = new SemaphoreLock(1);
      rLock_ = new NullLock();
   }
   
   /**
    * @see org.jboss.cache.lock.LockStrategy#readLock()
    */
   public Lock readLock()
   {
      return rLock_;
   }

   /**
    * @see org.jboss.cache.lock.LockStrategy#upgradeLockAttempt(long)
    */
   public Lock upgradeLockAttempt(long msecs) throws UpgradeException
   {
      // Since write is exclusive, we need to obtain the write lock first
      // before we can return the upgrade
      try {
         wLock_.tryLock(msecs, TimeUnit.MILLISECONDS);
      } catch (InterruptedException e) {
         throw new UpgradeException("Upgrade failed in " + msecs + " msecs", e);
      }
      return wLock_;
   }

   /**
    * @see org.jboss.cache.lock.LockStrategy#writeLock()
    */
   public Lock writeLock()
   {
      return wLock_;
   }
}
