package org.jboss.cache.invalidation;

import org.jboss.cache.CacheSPI;
import org.jboss.cache.DefaultCacheFactory;
import org.jboss.cache.Fqn;
import org.jboss.cache.util.TestingUtil;
import org.jboss.cache.config.Configuration;
import org.jboss.cache.config.EvictionConfig;
import org.jboss.cache.config.EvictionRegionConfig;
import org.jboss.cache.eviction.FIFOConfiguration;
import org.jboss.cache.transaction.DummyTransactionManagerLookup;
import org.testng.annotations.AfterMethod;
import org.testng.annotations.BeforeMethod;
import org.testng.annotations.Test;
import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.util.ArrayList;
import java.util.List;

/**
 * Make sure tombstones are evicted
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani</a>
 * @since 2.1.0
 */
@Test(groups = {"functional"})
public class TombstoneEvictionTest
{
   private CacheSPI c1, c2;
   private Fqn fqn = Fqn.fromString("/data/test");
   private Fqn dummy = Fqn.fromString("/data/dummy");
   private long evictionWaitTime = 4100;

   @BeforeMethod
   public void setUp() throws Exception
   {
      log.trace("**** setup called");
      c1 = (CacheSPI) new DefaultCacheFactory().createCache(false);

      // the FIFO policy cfg
      FIFOConfiguration cfg = new FIFOConfiguration();
      cfg.setMaxNodes(1);
      cfg.setMinTimeToLiveSeconds(0);

      // the region configuration
      EvictionRegionConfig regionCfg = new EvictionRegionConfig();
      regionCfg.setRegionFqn(dummy.getParent());
      regionCfg.setRegionName(dummy.getParent().toString());
      regionCfg.setEvictionPolicyConfig(cfg);

      // set regions in a list
      List<EvictionRegionConfig> evictionRegionConfigs = new ArrayList<EvictionRegionConfig>();
      evictionRegionConfigs.add(regionCfg);


      EvictionConfig ec = new EvictionConfig();
      ec.setWakeupIntervalSeconds(2);
      ec.setEvictionRegionConfigs(evictionRegionConfigs);

      c1.getConfiguration().setCacheMode(Configuration.CacheMode.INVALIDATION_SYNC);
      c1.getConfiguration().setNodeLockingScheme(Configuration.NodeLockingScheme.OPTIMISTIC);
      c1.getConfiguration().setTransactionManagerLookupClass(DummyTransactionManagerLookup.class.getName());
      c1.getConfiguration().setEvictionConfig(ec);

      c2 = (CacheSPI) new DefaultCacheFactory().createCache(c1.getConfiguration().clone(), false);

      c1.start();
      c2.start();

      TestingUtil.blockUntilViewsReceived(60000, c1, c2);
   }

   @AfterMethod
   public void tearDown()
   {
      TestingUtil.killCaches(c1, c2);
   }

   private static final Log log = LogFactory.getLog(TombstoneEvictionTest.class);

   public void testControl()
   {
      log.trace("***** entered testControl()");
      c1.put(fqn, "k", "v");
      c1.put(dummy, "k", "v");
      log.trace("***** nodes were added testControl()");


      assert c1.peek(fqn, false, true) != null : "Node should exist";
      assert c1.peek(dummy, false, true) != null : "Node should exist";
      log.trace("***** testControl() right before sleeping");

      TestingUtil.sleepThread(evictionWaitTime);
      log.trace("***** testControl() right after sleeping");

      assert c1.peek(fqn, false, true) == null : "Should have evicted";
      assert c1.peek(dummy, false, true) != null : "Node should exist";
   }

   public void testWithInvalidationMarkers()
   {
      log.trace(" **** testWithInvalidationMarkers() before put");
      c1.put(fqn, "k", "v");
      c1.put(dummy, "k", "v");
      log.trace(" **** testWithInvalidationMarkers() after put");

      assert c1.peek(fqn, false, true) != null : "Node should exist";
      assert c1.peek(dummy, false, true) != null : "Node should exist";

      assert c2.peek(fqn, false, true) != null : "Node should exist";
      assert c2.peek(dummy, false, true) != null : "Node should exist";

      TestingUtil.sleepThread(evictionWaitTime);

      assert c1.peek(fqn, false, true) == null : "Should have evicted";
      assert c1.peek(dummy, false, true) != null : "Node should exist";

      assert c2.peek(fqn, false, true) == null : "Should have evicted";
      assert c2.peek(dummy, false, true) != null : "Node should exist";
   }

   public void testWithTombstones()
   {
      c1.put(fqn, "k", "v");
      c1.removeNode(fqn);
      c1.put(dummy, "k", "v");

      assert c1.peek(fqn, false, true) != null : "Node should exist";
      assert c1.peek(dummy, false, true) != null : "Node should exist";

      assert c2.peek(fqn, false, true) != null : "Node should exist";
      assert c2.peek(dummy, false, true) != null : "Node should exist";

      TestingUtil.sleepThread(evictionWaitTime);

      assert c1.peek(fqn, false, true) == null : "Should have evicted";
      assert c1.peek(dummy, false, true) != null : "Node should exist";

      assert c2.peek(fqn, false, true) == null : "Should have evicted";
      assert c2.peek(dummy, false, true) != null : "Node should exist";
   }
}
