Working with Servers
====================

A *server* is a virtual machine instance that is managed by OpenStack Nova,
the `Compute` service. One advantage of using OpenStack is that the
virtualization layer makes it easy to create and discard servers as needed.

The `Server` object, generated by the `Compute::Server()` method, is used
to work with servers. You can create, update, or delete servers, and you can
perform various *actions* on the server, such as creating a backup image,
resizing it, or rebooting it.

To create an empty server:

    $serv = $compute->Server(); // assumes that $compute exists

To retrieve the data on an existing server:

    $serv = $compute->Server('9bfd203a-0695-410d-99a9-66c4194c967b');

## Creating, updating, or deleting a server

### Creating a new server

A server requires both a [Flavor object](flavors.md) and an
[Image object](images.md) to
be created. In addition, a server requires a name. You can easily create a
new server by setting the proper values and calling the `Create()` method:

    $server = $compute->Server();
    $server->name = 'Small Server';
    $server->flavor = $compute->Flavor(2);
    $server->image = $compute->Image('c195ef3b-9195-4474-b6f7-16e5bd86acd0');
    $server->Create();

As a shortcut, you can also pass these parameters to the `Create()` method
in an associative array:

    $server = $compute->Server();
    $server->Create(array(
        'name' => 'Small Server',
        'flavor' => $compute->Flavor(2),
        'image' => $compute->Image('c195ef3b-9195-4474-b6f7-16e5bd86acd0')));

Note that when the `Create()` request completes, the server is not actually
created. Instead, the completion of the request indicates that the server
build has been initiated in the Compute instance. Server builds typically
take 1-5 minutes to complete (depending upon the size of the server). However,
the initial response will return the server's ID as well as the assigned
root password:

    // assuming we've created $server, above
    printf("ID is %s, root password is %s\n",
        $server->id, $server->adminPass);

(Note: it is not recommended that you print out the root password because of
security risks. This is only provided as an example.)

When you create a new server on the Rackspace public cloud, you can also
associate it with one or more isolated networks. For more information, see
[Working with Cloud Networks](networks.md).

### Rebuilding an existing server

"Rebuilding" a server is nearly identical to creating one; you must supply
an Image object and a Flavor object, and the server is rebuilt using those
parameters. You can also change the server's name as part of the rebuild.
The primary difference between a create and a rebuild is that, in the rebuild,
the server's IP address(es) are retained (when the server is created, new IP
addresses are assigned).

To rebuild a server:

    $server = $compute->Server('abaf0...');     // existing server
    $server->Rebuild(array(
        'name' => 'A Bigger Server', // name is not required on rebuild
        'flavor' => $compute->Flavor(4),
        'image' => $compute->Image('c195ef3b-9195-4474-b6f7-16e5bd86acd0')));

### Updating a server

The `Update()` method is very similar to `Create()` except that the only
attributes of a server that you are permitted to update are its name and
the [access IP addresses](accessip.md).

    $compute = $cloud->Compute();
    $server = $compute->Server('908c5617-26c2-4535-99a9-3f20e4b74835');
    $server->Update(array('accessIPv4'=>'50.57.94.244'));

### Deleting a server

The `Delete()` method is short and sweet: it immediately initiates the
deletion of the server. Note that this is a destructive mechanism and it is
unlikely that the server can be recovered. Example:

    $compute = $cloud->Compute();
    $server = $compute->Server('908c5617-26c2-4535-99a9-3f20e4b74835');
    $server->Delete();  // BAM! bye-bye

## Server actions

You can perform various actions on a server, such as rebooting it, resizing
it, or changing the root password.

### Setting the root password

Use the `SetPassword()` method to change the root user's password:

    $server->SetPassword('new password');

Note that it may take a few second for the new password to take effect. Also,
password restrictions (such as the minimum number of characters, numbers of
punctuation characters, and so forth) are enforced by the operating system and are
not always detectable by the Compute service. This means that, even though
the `SetPassword()` method succeeds, the password may not be changed, and
there may not be any feedback to that effect.

### To reboot the server

You can perform either a *hard* reboot (this is like pulling the power cord
and then restarting) or a *soft* reboot (initiated by the operating system
and generally less disruptive than a hard reboot). A soft reboot is
performed by default:

    $server->Reboot();                      // soft reboot
    $server->Reboot(RAXSDK_SOFT_REBOOT);    // also a soft reboot
    $server->Reboot(RAXSDK_HARD_REBOOT);    // a hard reboot

If the server is "hung," or unresponsive, a hard reboot may sometimes be
the only way to access the server.

### To resize the server

A server can be resized by providing a new [Flavor object](flavors.md):

    $server->Resize($compute->Flavor(5));

Once the resize completes (check the `$server->status`), you can either
confirm it:

    $server->ResizeConfirm();

or revert it back to the original size:

    $server->ResizeRevert();

### To rescue/unrescue a server

In rescue mode, a server is rebuilt to a pristine state and the existing
filesystem is mounted so that you can edit files and diagnose issues.
See
[this document](http://docs.rackspace.com/servers/api/v2/cs-devguide/content/rescue_mode.html)
for more details.

Put server into rescue mode:

    $password = $server->Rescue();

The `$password` is the assigned root password of the rescue server.

Take server out of rescue mode:

    $server->Unrescue();

This restores the server to its original state (plus any changes you may have
made while it was in rescue mode).

## Volumes

See [Working with Volumes](volumes.md) for information on the 
`Server::AttachVolume()`
and `Server::DetachVolume()` methods. 

## What next?

* Return to the [Table of Contents](toc.md). 
* See also [Working with Networks](networks.md).
* To learn about dynamic 
  volume creation and assignment, see 
  [Working with Volumes](volumes.md).

