/*
 * @BEGIN LICENSE
 *
 * Psi4: an open-source quantum chemistry software package
 *
 * Copyright (c) 2007-2017 The Psi4 Developers.
 *
 * The copyrights for code used from other parties are included in
 * the corresponding files.
 *
 * This file is part of Psi4.
 *
 * Psi4 is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation, version 3.
 *
 * Psi4 is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along
 * with Psi4; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * @END LICENSE
 */

#include "psi4/libmints/vector.h"
#include "PBE_Cfunctional.h"
#include "utility.h"
#include <cmath>

using namespace psi;

namespace psi {

PBE_CFunctional::PBE_CFunctional()
{
    name_ = "PBE_C";
    description_ = "    PBE Correlation\n";
    citation_ = "    J.P. Perdew, K. Burke., and M. Ernzerhof, Phys. Rev. Lett., 77, 3865-3868, 1996\n";
    alpha_ = 1.0;
    omega_ = 0.0;
    lrc_ = false;
    gga_ = true;
    meta_ = false;
    parameters_["c"] =   6.2035049089939986E-01;
    parameters_["two_13"] =   1.2599210498948732E+00;
    parameters_["k"] =   3.0936677262801355E+00;
    parameters_["pi_m12"] =   5.6418958354775628E-01;
    parameters_["bet"] =   6.6724550603149205E-02;
    parameters_["gammas"] =   3.1090690869654901E-02;
    parameters_["d2fz0"] =   1.7099209341613653E+00;
    parameters_["Aa"] =   1.6886900000000000E-02;
    parameters_["a1a"] =   1.1125000000000000E-01;
    parameters_["b1a"] =   1.0356999999999999E+01;
    parameters_["b2a"] =   3.6231000000000000E+00;
    parameters_["b3a"] =   8.8026000000000004E-01;
    parameters_["b4a"] =   4.9670999999999998E-01;
    parameters_["c0p"] =   3.1090699999999999E-02;
    parameters_["a1p"] =   2.1370000000000000E-01;
    parameters_["b1p"] =   7.5956999999999999E+00;
    parameters_["b2p"] =   3.5876000000000001E+00;
    parameters_["b3p"] =   1.6382000000000001E+00;
    parameters_["b4p"] =   4.9293999999999999E-01;
    parameters_["c0f"] =   1.5545349999999999E-02;
    parameters_["a1f"] =   2.0548000000000000E-01;
    parameters_["b1f"] =   1.4118900000000000E+01;
    parameters_["b2f"] =   6.1977000000000002E+00;
    parameters_["b3f"] =   3.3662000000000001E+00;
    parameters_["b4f"] =   6.2517000000000000E-01;
}
PBE_CFunctional::~PBE_CFunctional()
{
}
void PBE_CFunctional::compute_functional(const std::map<std::string,SharedVector>& in, const std::map<std::string,SharedVector>& out, int npoints, int deriv, double alpha)
{
    double c = parameters_["c"];
    double two_13 = parameters_["two_13"];
    double k = parameters_["k"];
    double pi_m12 = parameters_["pi_m12"];
    double bet = parameters_["bet"];
    double gammas = parameters_["gammas"];
    double d2fz0 = parameters_["d2fz0"];
    double Aa = parameters_["Aa"];
    double a1a = parameters_["a1a"];
    double b1a = parameters_["b1a"];
    double b2a = parameters_["b2a"];
    double b3a = parameters_["b3a"];
    double b4a = parameters_["b4a"];
    double c0p = parameters_["c0p"];
    double a1p = parameters_["a1p"];
    double b1p = parameters_["b1p"];
    double b2p = parameters_["b2p"];
    double b3p = parameters_["b3p"];
    double b4p = parameters_["b4p"];
    double c0f = parameters_["c0f"];
    double a1f = parameters_["a1f"];
    double b1f = parameters_["b1f"];
    double b2f = parameters_["b2f"];
    double b3f = parameters_["b3f"];
    double b4f = parameters_["b4f"];

    // Overall scale factor
    double scale = alpha_ * alpha;

    // => Input variables <= //

    double* rho_ap = NULL;
    double* rho_bp = NULL;
    double* gamma_aap = NULL;
    double* gamma_abp = NULL;
    double* gamma_bbp = NULL;
    double* tau_ap = NULL;
    double* tau_bp = NULL;

    if (true) {
        rho_ap = in.find("RHO_A")->second->pointer();
        rho_bp = in.find("RHO_B")->second->pointer();
    }
    if (gga_) {
        gamma_aap = in.find("GAMMA_AA")->second->pointer();
        gamma_abp = in.find("GAMMA_AB")->second->pointer();
        gamma_bbp = in.find("GAMMA_BB")->second->pointer();
    }
    if (meta_)  {
        tau_ap = in.find("TAU_A")->second->pointer();
        tau_bp = in.find("TAU_B")->second->pointer();
    }

    // => Outut variables <= //

    double* v = NULL;

    double* v_rho_a = NULL;
    double* v_rho_b = NULL;
    double* v_gamma_aa = NULL;
    double* v_gamma_ab = NULL;
    double* v_gamma_bb = NULL;
    double* v_tau_a = NULL;
    double* v_tau_b = NULL;

    double* v_rho_a_rho_a = NULL;
    double* v_rho_a_rho_b = NULL;
    double* v_rho_b_rho_b = NULL;
    double* v_gamma_aa_gamma_aa = NULL;
    double* v_gamma_aa_gamma_ab = NULL;
    double* v_gamma_aa_gamma_bb = NULL;
    double* v_gamma_ab_gamma_ab = NULL;
    double* v_gamma_ab_gamma_bb = NULL;
    double* v_gamma_bb_gamma_bb = NULL;
    double* v_tau_a_tau_a = NULL;
    double* v_tau_a_tau_b = NULL;
    double* v_tau_b_tau_b = NULL;
    double* v_rho_a_gamma_aa = NULL;
    double* v_rho_a_gamma_ab = NULL;
    double* v_rho_a_gamma_bb = NULL;
    double* v_rho_b_gamma_aa = NULL;
    double* v_rho_b_gamma_ab = NULL;
    double* v_rho_b_gamma_bb = NULL;
    double* v_rho_a_tau_a = NULL;
    double* v_rho_a_tau_b = NULL;
    double* v_rho_b_tau_a = NULL;
    double* v_rho_b_tau_b = NULL;
    double* v_gamma_aa_tau_a = NULL;
    double* v_gamma_aa_tau_b = NULL;
    double* v_gamma_ab_tau_a = NULL;
    double* v_gamma_ab_tau_b = NULL;
    double* v_gamma_bb_tau_a = NULL;
    double* v_gamma_bb_tau_b = NULL;

    if (deriv >= 0) {
        v = out.find("V")->second->pointer();
    }
    if (deriv >= 1) {
        if (true) {
            v_rho_a = out.find("V_RHO_A")->second->pointer();
            v_rho_b = out.find("V_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa = out.find("V_GAMMA_AA")->second->pointer();
            v_gamma_ab = out.find("V_GAMMA_AB")->second->pointer();
            v_gamma_bb = out.find("V_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a = out.find("V_TAU_A")->second->pointer();
            v_tau_b = out.find("V_TAU_B")->second->pointer();
        }
    }
    if (deriv >= 2) {
        if (true) {
            v_rho_a_rho_a = out.find("V_RHO_A_RHO_A")->second->pointer();
            v_rho_a_rho_b = out.find("V_RHO_A_RHO_B")->second->pointer();
            v_rho_b_rho_b = out.find("V_RHO_B_RHO_B")->second->pointer();
        }
        if (gga_) {
            v_gamma_aa_gamma_aa = out.find("V_GAMMA_AA_GAMMA_AA")->second->pointer();
            v_gamma_aa_gamma_ab = out.find("V_GAMMA_AA_GAMMA_AB")->second->pointer();
            v_gamma_aa_gamma_bb = out.find("V_GAMMA_AA_GAMMA_BB")->second->pointer();
            v_gamma_ab_gamma_ab = out.find("V_GAMMA_AB_GAMMA_AB")->second->pointer();
            v_gamma_ab_gamma_bb = out.find("V_GAMMA_AB_GAMMA_BB")->second->pointer();
            v_gamma_bb_gamma_bb = out.find("V_GAMMA_BB_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_tau_a_tau_a = out.find("V_TAU_A_TAU_A")->second->pointer();
            v_tau_a_tau_b = out.find("V_TAU_A_TAU_B")->second->pointer();
            v_tau_b_tau_b = out.find("V_TAU_B_TAU_B")->second->pointer();
        }
        if (gga_) {
            v_rho_a_gamma_aa = out.find("V_RHO_A_GAMMA_AA")->second->pointer();
            v_rho_a_gamma_ab = out.find("V_RHO_A_GAMMA_AB")->second->pointer();
            v_rho_a_gamma_bb = out.find("V_RHO_A_GAMMA_BB")->second->pointer();
            v_rho_b_gamma_aa = out.find("V_RHO_B_GAMMA_AA")->second->pointer();
            v_rho_b_gamma_ab = out.find("V_RHO_B_GAMMA_AB")->second->pointer();
            v_rho_b_gamma_bb = out.find("V_RHO_B_GAMMA_BB")->second->pointer();
        }
        if (meta_) {
            v_rho_a_tau_a = out.find("V_RHO_A_TAU_A")->second->pointer();
            v_rho_a_tau_b = out.find("V_RHO_A_TAU_B")->second->pointer();
            v_rho_b_tau_a = out.find("V_RHO_B_TAU_A")->second->pointer();
            v_rho_b_tau_b = out.find("V_RHO_B_TAU_B")->second->pointer();
        }
        if (gga_ && meta_) {
            v_gamma_aa_tau_a = out.find("V_GAMMA_AA_TAU_A")->second->pointer();
            v_gamma_aa_tau_b = out.find("V_GAMMA_AA_TAU_B")->second->pointer();
            v_gamma_ab_tau_a = out.find("V_GAMMA_AB_TAU_A")->second->pointer();
            v_gamma_ab_tau_b = out.find("V_GAMMA_AB_TAU_B")->second->pointer();
            v_gamma_bb_tau_a = out.find("V_GAMMA_BB_TAU_A")->second->pointer();
            v_gamma_bb_tau_b = out.find("V_GAMMA_BB_TAU_B")->second->pointer();
        }
    }

    // => Loop over points <= //

    for (int Q = 0; Q < npoints; Q++) {

        // Input variables
        double rho_a;
        double rho_b;
        double gamma_aa;
        double gamma_ab;
        double gamma_bb;
        double tau_a;
        double tau_b;

        if (true) {
            rho_a = rho_ap[Q];
            rho_b = rho_bp[Q];
        }
        if (gga_) {
            gamma_aa = gamma_aap[Q];
            gamma_ab = gamma_abp[Q];
            gamma_bb = gamma_bbp[Q];
        }
        if (meta_) {
            tau_a = tau_ap[Q];
            tau_b = tau_bp[Q];
        }

        // Definitions (asymptotics to prevent numerical problems)
        if (rho_a < lsda_cutoff_ && rho_b < lsda_cutoff_) {
            continue;
        } else if (rho_a < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t16339 = pow(2.0,2.0/3.0);
                double t16340 = 1.0/gammas;
                double t16341 = 1.0/k;
                double t16342 = 1.0/(pi_m12*pi_m12);
                double t16343 = 1.0/pow(rho_b,7.0/3.0);
                double t16344 = 1.0/pow(rho_b,1.0/3.0);
                double t16345 = c*t16344;
                double t16346 = sqrt(t16345);
                double t16347 = pow(t16345,3.0/2.0);
                double t16348 = c*c;
                double t16349 = 1.0/pow(rho_b,2.0/3.0);
                double t16350 = 1.0/c0p;
                double t16351 = b1p*t16346;
                double t16352 = b3p*t16347;
                double t16353 = b4p*t16348*t16349;
                double t16354 = b2p*c*t16344;
                double t16355 = t16351+t16352+t16353+t16354;
                double t16356 = 1.0/t16355;
                double t16357 = t16350*t16356*(1.0/2.0);
                double t16358 = t16357+1.0;
                double t16359 = log(t16358);
                double t16360 = a1p*c*t16344;
                double t16361 = t16360+1.0;
                double t16362 = gamma_ab*2.0;
                double t16363 = gamma_bb+t16362;
                double t16364 = pow(2.0,1.0/3.0);
                double t16365 = t16364*2.0;
                double t16366 = t16365-2.0;
                double t16367 = two_13*2.0;
                double t16368 = t16367-2.0;
                double t16369 = 1.0/t16368;
                double t16370 = 1.0/c0f;
                double t16371 = b1f*t16346;
                double t16372 = b3f*t16347;
                double t16373 = b4f*t16348*t16349;
                double t16374 = b2f*c*t16344;
                double t16375 = t16371+t16372+t16373+t16374;
                double t16376 = 1.0/t16375;
                double t16377 = t16370*t16376*(1.0/2.0);
                double t16378 = t16377+1.0;
                double t16379 = log(t16378);
                double t16380 = a1f*c*t16344;
                double t16381 = t16380+1.0;
                double t16382 = c0f*t16381*t16379*2.0;
                double t16383 = c0p*t16361*t16359*2.0;
                double t16384 = t16382-t16383;
                double t16385 = t16366*t16384*t16369;
                double t16386 = t16383+t16385;
                double t16387 = t16340*t16386*2.0;
                double t16388 = exp(t16387);
                double t16389 = t16388-1.0;
                double t16390 = 1.0/t16389;
                double t16391 = bet*t16340*t16341*t16342*t16343*t16363*t16390*t16339*(1.0/1.6E1);
                v[Q] += scale * (-(rho_a+rho_b)*(t16383+t16385-gammas*log((bet*t16340*t16341*t16342*t16343*t16363*t16339*(t16391+1.0)*(1.0/1.6E1))/(t16391+(bet*bet)*1.0/(gammas*gammas)*1.0/(k*k)*1.0/(pi_m12*pi_m12*pi_m12*pi_m12)*1.0/pow(rho_b,1.4E1/3.0)*(t16363*t16363)*t16364*1.0/(t16389*t16389)*(1.0/1.28E2)+1.0)+1.0)*(1.0/2.0)));
            }

            // v_rho_a
            if (deriv >= 1) {
                double t16393 = pow(2.0,2.0/3.0);
                double t16394 = 1.0/gammas;
                double t16395 = 1.0/k;
                double t16396 = 1.0/(pi_m12*pi_m12);
                double t16397 = 1.0/pow(rho_b,7.0/3.0);
                double t16398 = 1.0/pow(rho_b,1.0/3.0);
                double t16399 = c*t16398;
                double t16400 = sqrt(t16399);
                double t16401 = pow(t16399,3.0/2.0);
                double t16402 = c*c;
                double t16403 = 1.0/pow(rho_b,2.0/3.0);
                double t16404 = 1.0/c0p;
                double t16405 = b1p*t16400;
                double t16406 = b3p*t16401;
                double t16407 = b4p*t16402*t16403;
                double t16408 = b2p*c*t16398;
                double t16409 = t16405+t16406+t16407+t16408;
                double t16410 = 1.0/t16409;
                double t16411 = t16410*t16404*(1.0/2.0);
                double t16412 = t16411+1.0;
                double t16413 = log(t16412);
                double t16414 = a1p*c*t16398;
                double t16415 = t16414+1.0;
                double t16416 = gamma_ab*2.0;
                double t16417 = gamma_bb+t16416;
                double t16418 = pow(2.0,1.0/3.0);
                double t16419 = t16418*2.0;
                double t16420 = t16419-2.0;
                double t16421 = two_13*2.0;
                double t16422 = t16421-2.0;
                double t16423 = 1.0/t16422;
                double t16424 = 1.0/c0f;
                double t16425 = b1f*t16400;
                double t16426 = b3f*t16401;
                double t16427 = b4f*t16402*t16403;
                double t16428 = b2f*c*t16398;
                double t16429 = t16425+t16426+t16427+t16428;
                double t16430 = 1.0/t16429;
                double t16431 = t16430*t16424*(1.0/2.0);
                double t16432 = t16431+1.0;
                double t16433 = log(t16432);
                double t16434 = a1f*c*t16398;
                double t16435 = t16434+1.0;
                double t16436 = c0f*t16433*t16435*2.0;
                double t16437 = c0p*t16413*t16415*2.0;
                double t16438 = t16436-t16437;
                double t16439 = t16420*t16423*t16438;
                double t16440 = t16437+t16439;
                double t16441 = t16440*t16394*2.0;
                double t16442 = exp(t16441);
                double t16443 = t16442-1.0;
                v_rho_a[Q] += scale * (-t16437-t16439+gammas*log((bet*t16417*t16393*t16394*t16395*t16396*t16397*((bet*t16417*t16393*t16394*t16395*t16396*t16397*(1.0/1.6E1))/(exp(t16394*(t16437+t16420*t16423*(t16436-c0p*t16413*t16415*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1))/((bet*t16417*t16393*t16394*t16395*t16396*t16397*(1.0/1.6E1))/t16443+(bet*bet)*1.0/(gammas*gammas)*1.0/(k*k)*1.0/(pi_m12*pi_m12*pi_m12*pi_m12)*1.0/pow(rho_b,1.4E1/3.0)*1.0/(t16443*t16443)*(t16417*t16417)*t16418*(1.0/1.28E2)+1.0)+1.0)*(1.0/2.0));
            }

            // v_rho_b
            if (deriv >= 1) {
                double t16445 = pow(2.0,2.0/3.0);
                double t16446 = 1.0/gammas;
                double t16447 = 1.0/k;
                double t16448 = 1.0/(pi_m12*pi_m12);
                double t16449 = 1.0/pow(rho_b,7.0/3.0);
                double t16450 = 1.0/pow(rho_b,1.0/3.0);
                double t16451 = c*t16450;
                double t16452 = sqrt(t16451);
                double t16453 = pow(t16451,3.0/2.0);
                double t16454 = c*c;
                double t16455 = 1.0/pow(rho_b,2.0/3.0);
                double t16456 = 1.0/c0p;
                double t16457 = b1p*t16452;
                double t16458 = b3p*t16453;
                double t16459 = b4p*t16454*t16455;
                double t16460 = b2p*c*t16450;
                double t16461 = t16460+t16457+t16458+t16459;
                double t16462 = 1.0/t16461;
                double t16463 = t16462*t16456*(1.0/2.0);
                double t16464 = t16463+1.0;
                double t16465 = log(t16464);
                double t16466 = a1p*c*t16450;
                double t16467 = t16466+1.0;
                double t16468 = gamma_ab*2.0;
                double t16469 = gamma_bb+t16468;
                double t16470 = pow(2.0,1.0/3.0);
                double t16471 = t16470*2.0;
                double t16472 = t16471-2.0;
                double t16473 = two_13*2.0;
                double t16474 = t16473-2.0;
                double t16475 = 1.0/t16474;
                double t16476 = 1.0/c0f;
                double t16477 = b1f*t16452;
                double t16478 = b3f*t16453;
                double t16479 = b4f*t16454*t16455;
                double t16480 = b2f*c*t16450;
                double t16481 = t16480+t16477+t16478+t16479;
                double t16482 = 1.0/t16481;
                double t16483 = t16482*t16476*(1.0/2.0);
                double t16484 = t16483+1.0;
                double t16485 = log(t16484);
                double t16486 = a1f*c*t16450;
                double t16487 = t16486+1.0;
                double t16488 = c0f*t16485*t16487*2.0;
                double t16489 = c0p*t16465*t16467*2.0;
                double t16490 = t16488-t16489;
                double t16491 = t16472*t16490*t16475;
                double t16492 = t16491+t16489;
                double t16493 = t16446*t16492*2.0;
                double t16494 = exp(t16493);
                double t16495 = t16494-1.0;
                double t16496 = 1.0/t16495;
                double t16497 = bet*t16445*t16446*t16447*t16448*t16449*t16469*t16496*(1.0/1.6E1);
                double t16498 = bet*bet;
                double t16499 = 1.0/(gammas*gammas);
                double t16500 = 1.0/(k*k);
                double t16501 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16502 = 1.0/pow(rho_b,1.4E1/3.0);
                double t16503 = 1.0/(t16495*t16495);
                double t16504 = t16469*t16469;
                double t16505 = t16500*t16501*t16502*t16503*t16504*t16470*t16498*t16499*(1.0/1.28E2);
                double t16506 = t16505+t16497+1.0;
                double t16507 = 1.0/t16506;
                double t16508 = 1.0/pow(rho_b,1.0E1/3.0);
                double t16509 = 1.0/pow(rho_b,4.0/3.0);
                double t16510 = 1.0/pow(rho_b,5.0/3.0);
                double t16511 = 1.0/sqrt(t16451);
                double t16512 = 1.0/t16464;
                double t16513 = 1.0/(t16461*t16461);
                double t16514 = b4p*t16510*t16454*(2.0/3.0);
                double t16515 = b2p*c*t16509*(1.0/3.0);
                double t16516 = b1p*c*t16511*t16509*(1.0/6.0);
                double t16517 = b3p*c*t16452*t16509*(1.0/2.0);
                double t16518 = t16514+t16515+t16516+t16517;
                double t16519 = t16512*t16513*t16518*t16467;
                double t16520 = t16497+1.0;
                double t16521 = bet*t16445*t16508*t16446*t16447*t16448*t16469*t16496*(7.0/4.8E1);
                double t16522 = 1.0/t16484;
                double t16523 = 1.0/(t16481*t16481);
                double t16524 = b4f*t16510*t16454*(2.0/3.0);
                double t16525 = b2f*c*t16509*(1.0/3.0);
                double t16526 = b1f*c*t16511*t16509*(1.0/6.0);
                double t16527 = b3f*c*t16452*t16509*(1.0/2.0);
                double t16528 = t16524+t16525+t16526+t16527;
                double t16529 = a1f*c*c0f*t16509*t16485*(2.0/3.0);
                double t16530 = a1p*c*c0p*t16509*t16465*(2.0/3.0);
                double t16531 = t16522*t16523*t16528*t16487;
                double t16532 = t16530+t16531-t16519-t16529;
                double t16533 = t16532*t16472*t16475;
                double t16534 = -t16530+t16533+t16519;
                double t16535 = bet*t16520*t16507*t16445*t16446*t16447*t16448*t16449*t16469*(1.0/1.6E1);
                double t16536 = t16535+1.0;
                v_rho_b[Q] += scale * (-t16491-t16489+gammas*log(t16536)*(1.0/2.0)-(rho_a+rho_b)*(-t16530+t16533+t16519+(gammas*(bet*t16507*t16445*t16446*t16447*t16448*t16449*t16469*(t16521+bet*t16503*t16445*t16447*t16448*t16449*t16494*t16469*t16499*(-t16530+t16533+t16519)*(1.0/8.0))*(1.0/1.6E1)+bet*t16520*t16507*t16445*t16508*t16446*t16447*t16448*t16469*(7.0/4.8E1)-bet*t16520*1.0/(t16506*t16506)*t16445*t16446*t16447*t16448*t16449*t16469*(t16521+1.0/pow(rho_b,1.7E1/3.0)*t16500*t16501*t16503*t16504*t16470*t16498*t16499*(7.0/1.92E2)+1.0/(gammas*gammas*gammas)*t16500*t16501*t16502*t16504*t16470*t16534*t16494*1.0/(t16495*t16495*t16495)*t16498*(1.0/3.2E1)+bet*t16503*t16534*t16445*t16447*t16448*t16449*t16494*t16469*t16499*(1.0/8.0))*(1.0/1.6E1))*(1.0/2.0))/t16536));
            }

            // v_gamma_ab
            if (deriv >= 1) {
                double t16539 = pow(2.0,2.0/3.0);
                double t16540 = 1.0/gammas;
                double t16541 = 1.0/k;
                double t16542 = 1.0/(pi_m12*pi_m12);
                double t16543 = 1.0/pow(rho_b,7.0/3.0);
                double t16544 = 1.0/pow(rho_b,1.0/3.0);
                double t16545 = c*t16544;
                double t16546 = sqrt(t16545);
                double t16547 = pow(t16545,3.0/2.0);
                double t16548 = c*c;
                double t16549 = 1.0/pow(rho_b,2.0/3.0);
                double t16550 = 1.0/c0p;
                double t16551 = b1p*t16546;
                double t16552 = b3p*t16547;
                double t16553 = b4p*t16548*t16549;
                double t16554 = b2p*c*t16544;
                double t16555 = t16551+t16552+t16553+t16554;
                double t16556 = 1.0/t16555;
                double t16557 = t16550*t16556*(1.0/2.0);
                double t16558 = t16557+1.0;
                double t16559 = log(t16558);
                double t16560 = a1p*c*t16544;
                double t16561 = t16560+1.0;
                double t16562 = gamma_ab*2.0;
                double t16563 = gamma_bb+t16562;
                double t16564 = pow(2.0,1.0/3.0);
                double t16565 = t16564*2.0;
                double t16566 = t16565-2.0;
                double t16567 = two_13*2.0;
                double t16568 = t16567-2.0;
                double t16569 = 1.0/t16568;
                double t16570 = 1.0/c0f;
                double t16571 = b1f*t16546;
                double t16572 = b3f*t16547;
                double t16573 = b4f*t16548*t16549;
                double t16574 = b2f*c*t16544;
                double t16575 = t16571+t16572+t16573+t16574;
                double t16576 = 1.0/t16575;
                double t16577 = t16570*t16576*(1.0/2.0);
                double t16578 = t16577+1.0;
                double t16579 = log(t16578);
                double t16580 = a1f*c*t16544;
                double t16581 = t16580+1.0;
                double t16582 = c0f*t16581*t16579*2.0;
                double t16583 = c0p*t16561*t16559*2.0;
                double t16584 = t16582-t16583;
                double t16585 = t16566*t16584*t16569;
                double t16586 = t16583+t16585;
                double t16587 = t16540*t16586*2.0;
                double t16588 = exp(t16587);
                double t16589 = t16588-1.0;
                double t16590 = 1.0/t16589;
                double t16591 = bet*t16540*t16541*t16542*t16543*t16563*t16590*t16539*(1.0/1.6E1);
                double t16592 = bet*bet;
                double t16593 = 1.0/(gammas*gammas);
                double t16594 = 1.0/(k*k);
                double t16595 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16596 = 1.0/pow(rho_b,1.4E1/3.0);
                double t16597 = 1.0/(t16589*t16589);
                double t16598 = t16563*t16563;
                double t16599 = t16564*t16592*t16593*t16594*t16595*t16596*t16597*t16598*(1.0/1.28E2);
                double t16600 = t16591+t16599+1.0;
                double t16601 = 1.0/t16600;
                double t16602 = t16591+1.0;
                v_gamma_ab[Q] += scale * ((gammas*(rho_a+rho_b)*(bet*t16601*t16602*t16540*t16541*t16542*t16543*t16539*(1.0/8.0)+t16601*t16563*t16590*t16564*t16592*t16593*t16594*t16595*t16596*(1.0/6.4E1)-bet*1.0/(t16600*t16600)*t16602*t16540*t16541*t16542*t16543*t16563*t16539*(bet*t16540*t16541*t16542*t16543*t16590*t16539*(1.0/8.0)+t16564*t16592*t16593*t16594*t16595*t16596*t16597*(gamma_ab*8.0+gamma_bb*4.0)*(1.0/1.28E2))*(1.0/1.6E1))*(1.0/2.0))/(bet*t16601*t16540*t16541*t16542*t16543*t16563*t16539*((bet*t16540*t16541*t16542*t16543*t16563*t16539*(1.0/1.6E1))/(exp(t16540*(t16583+t16566*t16569*(t16582-c0p*t16561*t16559*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1)+1.0));
            }

            // v_gamma_bb
            if (deriv >= 1) {
                double t16604 = pow(2.0,2.0/3.0);
                double t16605 = 1.0/gammas;
                double t16606 = 1.0/k;
                double t16607 = 1.0/(pi_m12*pi_m12);
                double t16608 = 1.0/pow(rho_b,7.0/3.0);
                double t16609 = 1.0/pow(rho_b,1.0/3.0);
                double t16610 = c*t16609;
                double t16611 = sqrt(t16610);
                double t16612 = pow(t16610,3.0/2.0);
                double t16613 = c*c;
                double t16614 = 1.0/pow(rho_b,2.0/3.0);
                double t16615 = 1.0/c0p;
                double t16616 = b1p*t16611;
                double t16617 = b3p*t16612;
                double t16618 = b4p*t16613*t16614;
                double t16619 = b2p*c*t16609;
                double t16620 = t16616+t16617+t16618+t16619;
                double t16621 = 1.0/t16620;
                double t16622 = t16621*t16615*(1.0/2.0);
                double t16623 = t16622+1.0;
                double t16624 = log(t16623);
                double t16625 = a1p*c*t16609;
                double t16626 = t16625+1.0;
                double t16627 = gamma_ab*2.0;
                double t16628 = gamma_bb+t16627;
                double t16629 = pow(2.0,1.0/3.0);
                double t16630 = t16629*2.0;
                double t16631 = t16630-2.0;
                double t16632 = two_13*2.0;
                double t16633 = t16632-2.0;
                double t16634 = 1.0/t16633;
                double t16635 = 1.0/c0f;
                double t16636 = b1f*t16611;
                double t16637 = b3f*t16612;
                double t16638 = b4f*t16613*t16614;
                double t16639 = b2f*c*t16609;
                double t16640 = t16636+t16637+t16638+t16639;
                double t16641 = 1.0/t16640;
                double t16642 = t16641*t16635*(1.0/2.0);
                double t16643 = t16642+1.0;
                double t16644 = log(t16643);
                double t16645 = a1f*c*t16609;
                double t16646 = t16645+1.0;
                double t16647 = c0f*t16644*t16646*2.0;
                double t16648 = c0p*t16624*t16626*2.0;
                double t16649 = t16647-t16648;
                double t16650 = t16631*t16634*t16649;
                double t16651 = t16650+t16648;
                double t16652 = t16605*t16651*2.0;
                double t16653 = exp(t16652);
                double t16654 = t16653-1.0;
                double t16655 = 1.0/t16654;
                double t16656 = bet*t16604*t16605*t16606*t16607*t16608*t16628*t16655*(1.0/1.6E1);
                double t16657 = bet*bet;
                double t16658 = 1.0/(gammas*gammas);
                double t16659 = 1.0/(k*k);
                double t16660 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16661 = 1.0/pow(rho_b,1.4E1/3.0);
                double t16662 = 1.0/(t16654*t16654);
                double t16663 = t16628*t16628;
                double t16664 = t16660*t16661*t16662*t16663*t16629*t16657*t16658*t16659*(1.0/1.28E2);
                double t16665 = t16664+t16656+1.0;
                double t16666 = 1.0/t16665;
                double t16667 = t16656+1.0;
                v_gamma_bb[Q] += scale * ((gammas*(rho_a+rho_b)*(bet*t16604*t16605*t16606*t16607*t16608*t16666*t16667*(1.0/1.6E1)+t16660*t16661*t16628*t16655*t16629*t16657*t16666*t16658*t16659*(1.0/1.28E2)-bet*t16604*t16605*t16606*t16607*t16608*t16628*1.0/(t16665*t16665)*t16667*(bet*t16604*t16605*t16606*t16607*t16608*t16655*(1.0/1.6E1)+t16660*t16661*t16662*t16629*t16657*t16658*t16659*(gamma_ab*4.0+gamma_bb*2.0)*(1.0/1.28E2))*(1.0/1.6E1))*(1.0/2.0))/(bet*t16604*t16605*t16606*t16607*t16608*t16628*t16666*((bet*t16604*t16605*t16606*t16607*t16608*t16628*(1.0/1.6E1))/(exp(t16605*(t16648+t16631*t16634*(t16647-c0p*t16624*t16626*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1)+1.0));
            }

        } else if (rho_b < lsda_cutoff_) {
            // v
            if (deriv >= 0) {
                double t16671 = pow(2.0,2.0/3.0);
                double t16672 = 1.0/gammas;
                double t16673 = 1.0/k;
                double t16674 = 1.0/(pi_m12*pi_m12);
                double t16675 = 1.0/pow(rho_a,7.0/3.0);
                double t16676 = 1.0/pow(rho_a,1.0/3.0);
                double t16677 = c*t16676;
                double t16678 = sqrt(t16677);
                double t16679 = pow(t16677,3.0/2.0);
                double t16680 = c*c;
                double t16681 = 1.0/pow(rho_a,2.0/3.0);
                double t16682 = 1.0/c0p;
                double t16683 = b1p*t16678;
                double t16684 = b3p*t16679;
                double t16685 = b4p*t16680*t16681;
                double t16686 = b2p*c*t16676;
                double t16687 = t16683+t16684+t16685+t16686;
                double t16688 = 1.0/t16687;
                double t16689 = t16682*t16688*(1.0/2.0);
                double t16690 = t16689+1.0;
                double t16691 = log(t16690);
                double t16692 = a1p*c*t16676;
                double t16693 = t16692+1.0;
                double t16694 = gamma_ab*2.0;
                double t16695 = gamma_aa+t16694;
                double t16696 = pow(2.0,1.0/3.0);
                double t16697 = t16696*2.0;
                double t16698 = t16697-2.0;
                double t16699 = two_13*2.0;
                double t16700 = t16699-2.0;
                double t16701 = 1.0/t16700;
                double t16702 = 1.0/c0f;
                double t16703 = b1f*t16678;
                double t16704 = b3f*t16679;
                double t16705 = b4f*t16680*t16681;
                double t16706 = b2f*c*t16676;
                double t16707 = t16703+t16704+t16705+t16706;
                double t16708 = 1.0/t16707;
                double t16709 = t16702*t16708*(1.0/2.0);
                double t16710 = t16709+1.0;
                double t16711 = log(t16710);
                double t16712 = a1f*c*t16676;
                double t16713 = t16712+1.0;
                double t16714 = c0f*t16711*t16713*2.0;
                double t16715 = c0p*t16691*t16693*2.0;
                double t16716 = t16714-t16715;
                double t16717 = t16701*t16716*t16698;
                double t16718 = t16715+t16717;
                double t16719 = t16672*t16718*2.0;
                double t16720 = exp(t16719);
                double t16721 = t16720-1.0;
                double t16722 = 1.0/t16721;
                double t16723 = bet*t16722*t16671*t16672*t16673*t16674*t16675*t16695*(1.0/1.6E1);
                v[Q] += scale * (-(rho_a+rho_b)*(t16715+t16717-gammas*log((bet*t16671*t16672*t16673*t16674*t16675*t16695*(t16723+1.0)*(1.0/1.6E1))/(t16723+(bet*bet)*1.0/(gammas*gammas)*1.0/(k*k)*1.0/(pi_m12*pi_m12*pi_m12*pi_m12)*1.0/pow(rho_a,1.4E1/3.0)*1.0/(t16721*t16721)*(t16695*t16695)*t16696*(1.0/1.28E2)+1.0)+1.0)*(1.0/2.0)));
            }

            // v_rho_a
            if (deriv >= 1) {
                double t16725 = pow(2.0,2.0/3.0);
                double t16726 = 1.0/gammas;
                double t16727 = 1.0/k;
                double t16728 = 1.0/(pi_m12*pi_m12);
                double t16729 = 1.0/pow(rho_a,7.0/3.0);
                double t16730 = 1.0/pow(rho_a,1.0/3.0);
                double t16731 = c*t16730;
                double t16732 = sqrt(t16731);
                double t16733 = pow(t16731,3.0/2.0);
                double t16734 = c*c;
                double t16735 = 1.0/pow(rho_a,2.0/3.0);
                double t16736 = 1.0/c0p;
                double t16737 = b1p*t16732;
                double t16738 = b3p*t16733;
                double t16739 = b4p*t16734*t16735;
                double t16740 = b2p*c*t16730;
                double t16741 = t16740+t16737+t16738+t16739;
                double t16742 = 1.0/t16741;
                double t16743 = t16742*t16736*(1.0/2.0);
                double t16744 = t16743+1.0;
                double t16745 = log(t16744);
                double t16746 = a1p*c*t16730;
                double t16747 = t16746+1.0;
                double t16748 = gamma_ab*2.0;
                double t16749 = gamma_aa+t16748;
                double t16750 = pow(2.0,1.0/3.0);
                double t16751 = t16750*2.0;
                double t16752 = t16751-2.0;
                double t16753 = two_13*2.0;
                double t16754 = t16753-2.0;
                double t16755 = 1.0/t16754;
                double t16756 = 1.0/c0f;
                double t16757 = b1f*t16732;
                double t16758 = b3f*t16733;
                double t16759 = b4f*t16734*t16735;
                double t16760 = b2f*c*t16730;
                double t16761 = t16760+t16757+t16758+t16759;
                double t16762 = 1.0/t16761;
                double t16763 = t16762*t16756*(1.0/2.0);
                double t16764 = t16763+1.0;
                double t16765 = log(t16764);
                double t16766 = a1f*c*t16730;
                double t16767 = t16766+1.0;
                double t16768 = c0f*t16765*t16767*2.0;
                double t16769 = c0p*t16745*t16747*2.0;
                double t16770 = t16768-t16769;
                double t16771 = t16752*t16770*t16755;
                double t16772 = t16771+t16769;
                double t16773 = t16726*t16772*2.0;
                double t16774 = exp(t16773);
                double t16775 = t16774-1.0;
                double t16776 = 1.0/t16775;
                double t16777 = bet*t16725*t16726*t16727*t16728*t16729*t16749*t16776*(1.0/1.6E1);
                double t16778 = bet*bet;
                double t16779 = 1.0/(gammas*gammas);
                double t16780 = 1.0/(k*k);
                double t16781 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16782 = 1.0/pow(rho_a,1.4E1/3.0);
                double t16783 = 1.0/(t16775*t16775);
                double t16784 = t16749*t16749;
                double t16785 = t16750*t16780*t16781*t16782*t16783*t16784*t16778*t16779*(1.0/1.28E2);
                double t16786 = t16785+t16777+1.0;
                double t16787 = 1.0/t16786;
                double t16788 = 1.0/pow(rho_a,1.0E1/3.0);
                double t16789 = 1.0/pow(rho_a,4.0/3.0);
                double t16790 = 1.0/pow(rho_a,5.0/3.0);
                double t16791 = 1.0/sqrt(t16731);
                double t16792 = 1.0/t16744;
                double t16793 = 1.0/(t16741*t16741);
                double t16794 = b4p*t16734*t16790*(2.0/3.0);
                double t16795 = b2p*c*t16789*(1.0/3.0);
                double t16796 = b1p*c*t16791*t16789*(1.0/6.0);
                double t16797 = b3p*c*t16732*t16789*(1.0/2.0);
                double t16798 = t16794+t16795+t16796+t16797;
                double t16799 = t16747*t16792*t16793*t16798;
                double t16800 = t16777+1.0;
                double t16801 = bet*t16725*t16726*t16727*t16728*t16749*t16776*t16788*(7.0/4.8E1);
                double t16802 = 1.0/t16764;
                double t16803 = 1.0/(t16761*t16761);
                double t16804 = b4f*t16734*t16790*(2.0/3.0);
                double t16805 = b2f*c*t16789*(1.0/3.0);
                double t16806 = b1f*c*t16791*t16789*(1.0/6.0);
                double t16807 = b3f*c*t16732*t16789*(1.0/2.0);
                double t16808 = t16804+t16805+t16806+t16807;
                double t16809 = a1f*c*c0f*t16765*t16789*(2.0/3.0);
                double t16810 = a1p*c*c0p*t16745*t16789*(2.0/3.0);
                double t16811 = t16802*t16803*t16808*t16767;
                double t16812 = t16810+t16811-t16809-t16799;
                double t16813 = t16812*t16752*t16755;
                double t16814 = -t16810+t16813+t16799;
                double t16815 = bet*t16800*t16725*t16726*t16727*t16728*t16729*t16749*t16787*(1.0/1.6E1);
                double t16816 = t16815+1.0;
                v_rho_a[Q] += scale * (-t16771-t16769+gammas*log(t16816)*(1.0/2.0)-(rho_a+rho_b)*(-t16810+t16813+t16799+(gammas*(bet*t16725*t16726*t16727*t16728*t16729*t16749*t16787*(t16801+bet*t16725*t16727*t16728*t16729*t16774*t16783*t16749*t16779*(-t16810+t16813+t16799)*(1.0/8.0))*(1.0/1.6E1)+bet*t16800*t16725*t16726*t16727*t16728*t16749*t16787*t16788*(7.0/4.8E1)-bet*t16800*t16725*t16726*t16727*t16728*t16729*t16749*1.0/(t16786*t16786)*(t16801+1.0/pow(rho_a,1.7E1/3.0)*t16750*t16780*t16781*t16783*t16784*t16778*t16779*(7.0/1.92E2)+1.0/(gammas*gammas*gammas)*t16750*t16814*t16780*t16781*t16782*t16774*1.0/(t16775*t16775*t16775)*t16784*t16778*(1.0/3.2E1)+bet*t16814*t16725*t16727*t16728*t16729*t16774*t16783*t16749*t16779*(1.0/8.0))*(1.0/1.6E1))*(1.0/2.0))/t16816));
            }

            // v_rho_b
            if (deriv >= 1) {
                double t16818 = pow(2.0,2.0/3.0);
                double t16819 = 1.0/gammas;
                double t16820 = 1.0/k;
                double t16821 = 1.0/(pi_m12*pi_m12);
                double t16822 = 1.0/pow(rho_a,7.0/3.0);
                double t16823 = 1.0/pow(rho_a,1.0/3.0);
                double t16824 = c*t16823;
                double t16825 = sqrt(t16824);
                double t16826 = pow(t16824,3.0/2.0);
                double t16827 = c*c;
                double t16828 = 1.0/pow(rho_a,2.0/3.0);
                double t16829 = 1.0/c0p;
                double t16830 = b1p*t16825;
                double t16831 = b3p*t16826;
                double t16832 = b4p*t16827*t16828;
                double t16833 = b2p*c*t16823;
                double t16834 = t16830+t16831+t16832+t16833;
                double t16835 = 1.0/t16834;
                double t16836 = t16835*t16829*(1.0/2.0);
                double t16837 = t16836+1.0;
                double t16838 = log(t16837);
                double t16839 = a1p*c*t16823;
                double t16840 = t16839+1.0;
                double t16841 = gamma_ab*2.0;
                double t16842 = gamma_aa+t16841;
                double t16843 = pow(2.0,1.0/3.0);
                double t16844 = t16843*2.0;
                double t16845 = t16844-2.0;
                double t16846 = two_13*2.0;
                double t16847 = t16846-2.0;
                double t16848 = 1.0/t16847;
                double t16849 = 1.0/c0f;
                double t16850 = b1f*t16825;
                double t16851 = b3f*t16826;
                double t16852 = b4f*t16827*t16828;
                double t16853 = b2f*c*t16823;
                double t16854 = t16850+t16851+t16852+t16853;
                double t16855 = 1.0/t16854;
                double t16856 = t16855*t16849*(1.0/2.0);
                double t16857 = t16856+1.0;
                double t16858 = log(t16857);
                double t16859 = a1f*c*t16823;
                double t16860 = t16859+1.0;
                double t16861 = c0f*t16860*t16858*2.0;
                double t16862 = c0p*t16840*t16838*2.0;
                double t16863 = t16861-t16862;
                double t16864 = t16845*t16863*t16848;
                double t16865 = t16862+t16864;
                double t16866 = t16819*t16865*2.0;
                double t16867 = exp(t16866);
                double t16868 = t16867-1.0;
                v_rho_b[Q] += scale * (-t16862-t16864+gammas*log((bet*t16820*t16821*t16822*t16842*t16818*t16819*((bet*t16820*t16821*t16822*t16842*t16818*t16819*(1.0/1.6E1))/(exp(t16819*(t16862+t16845*t16848*(t16861-c0p*t16840*t16838*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1))/((bet*t16820*t16821*t16822*t16842*t16818*t16819*(1.0/1.6E1))/t16868+(bet*bet)*1.0/(gammas*gammas)*1.0/(k*k)*1.0/(pi_m12*pi_m12*pi_m12*pi_m12)*1.0/pow(rho_a,1.4E1/3.0)*(t16842*t16842)*t16843*1.0/(t16868*t16868)*(1.0/1.28E2)+1.0)+1.0)*(1.0/2.0));
            }

            // v_gamma_aa
            if (deriv >= 1) {
                double t16870 = pow(2.0,2.0/3.0);
                double t16871 = 1.0/gammas;
                double t16872 = 1.0/k;
                double t16873 = 1.0/(pi_m12*pi_m12);
                double t16874 = 1.0/pow(rho_a,7.0/3.0);
                double t16875 = 1.0/pow(rho_a,1.0/3.0);
                double t16876 = c*t16875;
                double t16877 = sqrt(t16876);
                double t16878 = pow(t16876,3.0/2.0);
                double t16879 = c*c;
                double t16880 = 1.0/pow(rho_a,2.0/3.0);
                double t16881 = 1.0/c0p;
                double t16882 = b1p*t16877;
                double t16883 = b3p*t16878;
                double t16884 = b4p*t16880*t16879;
                double t16885 = b2p*c*t16875;
                double t16886 = t16882+t16883+t16884+t16885;
                double t16887 = 1.0/t16886;
                double t16888 = t16881*t16887*(1.0/2.0);
                double t16889 = t16888+1.0;
                double t16890 = log(t16889);
                double t16891 = a1p*c*t16875;
                double t16892 = t16891+1.0;
                double t16893 = gamma_ab*2.0;
                double t16894 = gamma_aa+t16893;
                double t16895 = pow(2.0,1.0/3.0);
                double t16896 = t16895*2.0;
                double t16897 = t16896-2.0;
                double t16898 = two_13*2.0;
                double t16899 = t16898-2.0;
                double t16900 = 1.0/t16899;
                double t16901 = 1.0/c0f;
                double t16902 = b1f*t16877;
                double t16903 = b3f*t16878;
                double t16904 = b4f*t16880*t16879;
                double t16905 = b2f*c*t16875;
                double t16906 = t16902+t16903+t16904+t16905;
                double t16907 = 1.0/t16906;
                double t16908 = t16901*t16907*(1.0/2.0);
                double t16909 = t16908+1.0;
                double t16910 = log(t16909);
                double t16911 = a1f*c*t16875;
                double t16912 = t16911+1.0;
                double t16913 = c0f*t16910*t16912*2.0;
                double t16914 = c0p*t16890*t16892*2.0;
                double t16915 = t16913-t16914;
                double t16916 = t16900*t16915*t16897;
                double t16917 = t16914+t16916;
                double t16918 = t16871*t16917*2.0;
                double t16919 = exp(t16918);
                double t16920 = t16919-1.0;
                double t16921 = 1.0/t16920;
                double t16922 = bet*t16921*t16870*t16871*t16872*t16873*t16874*t16894*(1.0/1.6E1);
                double t16923 = bet*bet;
                double t16924 = 1.0/(gammas*gammas);
                double t16925 = 1.0/(k*k);
                double t16926 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16927 = 1.0/pow(rho_a,1.4E1/3.0);
                double t16928 = 1.0/(t16920*t16920);
                double t16929 = t16894*t16894;
                double t16930 = t16923*t16924*t16925*t16926*t16927*t16928*t16929*t16895*(1.0/1.28E2);
                double t16931 = t16930+t16922+1.0;
                double t16932 = 1.0/t16931;
                double t16933 = t16922+1.0;
                v_gamma_aa[Q] += scale * ((gammas*(rho_a+rho_b)*(bet*t16932*t16870*t16933*t16871*t16872*t16873*t16874*(1.0/1.6E1)+t16921*t16923*t16932*t16924*t16925*t16926*t16927*t16894*t16895*(1.0/1.28E2)-bet*1.0/(t16931*t16931)*t16870*t16933*t16871*t16872*t16873*t16874*t16894*(bet*t16921*t16870*t16871*t16872*t16873*t16874*(1.0/1.6E1)+t16923*t16924*t16925*t16926*t16927*t16928*t16895*(gamma_aa*2.0+gamma_ab*4.0)*(1.0/1.28E2))*(1.0/1.6E1))*(1.0/2.0))/(bet*t16932*t16870*t16871*t16872*t16873*t16874*t16894*((bet*t16870*t16871*t16872*t16873*t16874*t16894*(1.0/1.6E1))/(exp(t16871*(t16914+t16900*t16897*(t16913-c0p*t16890*t16892*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1)+1.0));
            }

            // v_gamma_ab
            if (deriv >= 1) {
                double t16935 = pow(2.0,2.0/3.0);
                double t16936 = 1.0/gammas;
                double t16937 = 1.0/k;
                double t16938 = 1.0/(pi_m12*pi_m12);
                double t16939 = 1.0/pow(rho_a,7.0/3.0);
                double t16940 = 1.0/pow(rho_a,1.0/3.0);
                double t16941 = c*t16940;
                double t16942 = sqrt(t16941);
                double t16943 = pow(t16941,3.0/2.0);
                double t16944 = c*c;
                double t16945 = 1.0/pow(rho_a,2.0/3.0);
                double t16946 = 1.0/c0p;
                double t16947 = b1p*t16942;
                double t16948 = b3p*t16943;
                double t16949 = b4p*t16944*t16945;
                double t16950 = b2p*c*t16940;
                double t16951 = t16950+t16947+t16948+t16949;
                double t16952 = 1.0/t16951;
                double t16953 = t16952*t16946*(1.0/2.0);
                double t16954 = t16953+1.0;
                double t16955 = log(t16954);
                double t16956 = a1p*c*t16940;
                double t16957 = t16956+1.0;
                double t16958 = gamma_ab*2.0;
                double t16959 = gamma_aa+t16958;
                double t16960 = pow(2.0,1.0/3.0);
                double t16961 = t16960*2.0;
                double t16962 = t16961-2.0;
                double t16963 = two_13*2.0;
                double t16964 = t16963-2.0;
                double t16965 = 1.0/t16964;
                double t16966 = 1.0/c0f;
                double t16967 = b1f*t16942;
                double t16968 = b3f*t16943;
                double t16969 = b4f*t16944*t16945;
                double t16970 = b2f*c*t16940;
                double t16971 = t16970+t16967+t16968+t16969;
                double t16972 = 1.0/t16971;
                double t16973 = t16972*t16966*(1.0/2.0);
                double t16974 = t16973+1.0;
                double t16975 = log(t16974);
                double t16976 = a1f*c*t16940;
                double t16977 = t16976+1.0;
                double t16978 = c0f*t16975*t16977*2.0;
                double t16979 = c0p*t16955*t16957*2.0;
                double t16980 = t16978-t16979;
                double t16981 = t16962*t16980*t16965;
                double t16982 = t16981+t16979;
                double t16983 = t16936*t16982*2.0;
                double t16984 = exp(t16983);
                double t16985 = t16984-1.0;
                double t16986 = 1.0/t16985;
                double t16987 = bet*t16935*t16936*t16937*t16938*t16939*t16959*t16986*(1.0/1.6E1);
                double t16988 = bet*bet;
                double t16989 = 1.0/(gammas*gammas);
                double t16990 = 1.0/(k*k);
                double t16991 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16992 = 1.0/pow(rho_a,1.4E1/3.0);
                double t16993 = 1.0/(t16985*t16985);
                double t16994 = t16959*t16959;
                double t16995 = t16960*t16990*t16991*t16992*t16993*t16994*t16988*t16989*(1.0/1.28E2);
                double t16996 = t16995+t16987+1.0;
                double t16997 = 1.0/t16996;
                double t16998 = t16987+1.0;
                v_gamma_ab[Q] += scale * ((gammas*(rho_a+rho_b)*(bet*t16935*t16936*t16937*t16938*t16939*t16997*t16998*(1.0/8.0)+t16960*t16990*t16991*t16992*t16959*t16986*t16988*t16997*t16989*(1.0/6.4E1)-bet*t16935*t16936*t16937*t16938*t16939*t16959*1.0/(t16996*t16996)*t16998*(bet*t16935*t16936*t16937*t16938*t16939*t16986*(1.0/8.0)+t16960*t16990*t16991*t16992*t16993*t16988*t16989*(gamma_aa*4.0+gamma_ab*8.0)*(1.0/1.28E2))*(1.0/1.6E1))*(1.0/2.0))/(bet*t16935*t16936*t16937*t16938*t16939*t16959*t16997*((bet*t16935*t16936*t16937*t16938*t16939*t16959*(1.0/1.6E1))/(exp(t16936*(t16979+t16962*t16965*(t16978-c0p*t16955*t16957*2.0))*2.0)-1.0)+1.0)*(1.0/1.6E1)+1.0));
            }

        } else {
            // v
            if (deriv >= 0) {
                double t15623 = rho_a+rho_b;
                double t15624 = 1.0/pow(t15623,1.0/3.0);
                double t15625 = c*t15624;
                double t15626 = 1.0/gammas;
                double t15627 = 1.0/k;
                double t15628 = 1.0/(pi_m12*pi_m12);
                double t15629 = 1.0/pow(t15623,7.0/3.0);
                double t15630 = 1.0/t15623;
                double t15631 = rho_a-rho_b;
                double t15632 = t15630*t15631;
                double t15633 = 1.0/c0p;
                double t15634 = sqrt(t15625);
                double t15635 = b1p*t15634;
                double t15636 = pow(t15625,3.0/2.0);
                double t15637 = b3p*t15636;
                double t15638 = c*c;
                double t15639 = 1.0/pow(t15623,2.0/3.0);
                double t15640 = b4p*t15638*t15639;
                double t15641 = b2p*c*t15624;
                double t15642 = t15640+t15641+t15635+t15637;
                double t15643 = 1.0/t15642;
                double t15644 = t15633*t15643*(1.0/2.0);
                double t15645 = t15644+1.0;
                double t15646 = log(t15645);
                double t15647 = a1p*c*t15624;
                double t15648 = t15647+1.0;
                double t15649 = c0p*t15646*t15648*2.0;
                double t15650 = t15631*t15631;
                double t15651 = t15632+1.0;
                double t15652 = -t15632+1.0;
                double t15653 = two_13*2.0;
                double t15654 = t15653-2.0;
                double t15655 = 1.0/t15654;
                double t15656 = 1.0/(t15623*t15623*t15623*t15623);
                double t15657 = t15650*t15650;
                double t15658 = pow(t15651,4.0/3.0);
                double t15659 = pow(t15652,4.0/3.0);
                double t15660 = t15658+t15659-2.0;
                double t15661 = pow(t15651,2.0/3.0);
                double t15662 = t15661*(1.0/2.0);
                double t15663 = pow(t15652,2.0/3.0);
                double t15664 = t15663*(1.0/2.0);
                double t15665 = t15662+t15664;
                double t15666 = 1.0/(t15665*t15665);
                double t15667 = 1.0/(t15665*t15665*t15665);
                double t15668 = 1.0/c0f;
                double t15669 = b1f*t15634;
                double t15670 = b3f*t15636;
                double t15671 = b4f*t15638*t15639;
                double t15672 = b2f*c*t15624;
                double t15673 = t15670+t15671+t15672+t15669;
                double t15674 = 1.0/t15673;
                double t15675 = t15674*t15668*(1.0/2.0);
                double t15676 = t15675+1.0;
                double t15677 = log(t15676);
                double t15678 = a1f*c*t15624;
                double t15679 = t15678+1.0;
                double t15705 = c0f*t15677*t15679*2.0;
                double t15680 = -t15705+t15649;
                double t15681 = 1.0/d2fz0;
                double t15682 = 1.0/Aa;
                double t15683 = b1a*t15634;
                double t15684 = b3a*t15636;
                double t15685 = b4a*t15638*t15639;
                double t15686 = b2a*c*t15624;
                double t15687 = t15683+t15684+t15685+t15686;
                double t15688 = 1.0/t15687;
                double t15689 = t15682*t15688*(1.0/2.0);
                double t15690 = t15689+1.0;
                double t15691 = log(t15690);
                double t15692 = a1a*c*t15624;
                double t15693 = t15692+1.0;
                double t15694 = t15656*t15657;
                double t15695 = t15694-1.0;
                double t15696 = Aa*t15660*t15681*t15655*t15691*t15693*t15695*2.0;
                double t15697 = t15649+t15696-t15660*t15680*t15655*t15656*t15657;
                double t15698 = t15626*t15667*t15697;
                double t15699 = exp(t15698);
                double t15700 = t15699-1.0;
                double t15701 = 1.0/t15700;
                double t15702 = gamma_ab*2.0;
                double t15703 = gamma_aa+gamma_bb+t15702;
                double t15704 = bet*t15701*t15703*t15626*t15627*t15628*t15629*t15666*(1.0/1.6E1);
                double t15706 = t15665*t15665;
                double t15707 = t15705-t15649;
                double t15708 = t15660*t15707*t15655*t15656*t15657;
                v[Q] += scale * (-t15623*(t15708+t15649+t15696-gammas*t15706*t15665*log((bet*t15703*t15626*t15627*t15628*t15629*t15666*(t15704+1.0)*(1.0/1.6E1))/(t15704+(bet*bet)*1.0/(gammas*gammas)*1.0/(k*k)*1.0/(pi_m12*pi_m12*pi_m12*pi_m12)*(t15703*t15703)*1.0/pow(t15623,1.4E1/3.0)*1.0/(t15665*t15665*t15665*t15665)*1.0/pow(exp(t15626*t15667*(t15708+t15649+t15696))-1.0,2.0)*(1.0/2.56E2)+1.0)+1.0)));
            }

            // v_rho_a
            if (deriv >= 1) {
                double t15710 = rho_a+rho_b;
                double t15711 = 1.0/gammas;
                double t15712 = 1.0/k;
                double t15713 = 1.0/(pi_m12*pi_m12);
                double t15714 = 1.0/pow(t15710,7.0/3.0);
                double t15715 = 1.0/t15710;
                double t15716 = rho_a-rho_b;
                double t15717 = t15715*t15716;
                double t15718 = 1.0/pow(t15710,1.0/3.0);
                double t15719 = c*t15718;
                double t15720 = sqrt(t15719);
                double t15721 = pow(t15719,3.0/2.0);
                double t15722 = c*c;
                double t15723 = 1.0/pow(t15710,2.0/3.0);
                double t15724 = 1.0/c0p;
                double t15725 = b1p*t15720;
                double t15726 = b3p*t15721;
                double t15727 = b4p*t15722*t15723;
                double t15728 = b2p*c*t15718;
                double t15729 = t15725+t15726+t15727+t15728;
                double t15730 = 1.0/t15729;
                double t15731 = t15730*t15724*(1.0/2.0);
                double t15732 = t15731+1.0;
                double t15733 = log(t15732);
                double t15734 = a1p*c*t15718;
                double t15735 = t15734+1.0;
                double t15736 = c0p*t15733*t15735*2.0;
                double t15737 = t15716*t15716;
                double t15738 = t15717+1.0;
                double t15739 = -t15717+1.0;
                double t15740 = two_13*2.0;
                double t15741 = t15740-2.0;
                double t15742 = 1.0/t15741;
                double t15743 = 1.0/(t15710*t15710*t15710*t15710);
                double t15744 = t15737*t15737;
                double t15745 = pow(t15738,4.0/3.0);
                double t15746 = pow(t15739,4.0/3.0);
                double t15747 = t15745+t15746-2.0;
                double t15748 = pow(t15738,2.0/3.0);
                double t15749 = t15748*(1.0/2.0);
                double t15750 = pow(t15739,2.0/3.0);
                double t15751 = t15750*(1.0/2.0);
                double t15752 = t15751+t15749;
                double t15753 = 1.0/(t15752*t15752);
                double t15754 = 1.0/(t15752*t15752*t15752);
                double t15755 = 1.0/c0f;
                double t15756 = b1f*t15720;
                double t15757 = b3f*t15721;
                double t15758 = b4f*t15722*t15723;
                double t15759 = b2f*c*t15718;
                double t15760 = t15756+t15757+t15758+t15759;
                double t15761 = 1.0/t15760;
                double t15762 = t15761*t15755*(1.0/2.0);
                double t15763 = t15762+1.0;
                double t15764 = log(t15763);
                double t15765 = a1f*c*t15718;
                double t15766 = t15765+1.0;
                double t15792 = c0f*t15764*t15766*2.0;
                double t15767 = t15736-t15792;
                double t15768 = 1.0/d2fz0;
                double t15769 = 1.0/Aa;
                double t15770 = b1a*t15720;
                double t15771 = b3a*t15721;
                double t15772 = b4a*t15722*t15723;
                double t15773 = b2a*c*t15718;
                double t15774 = t15770+t15771+t15772+t15773;
                double t15775 = 1.0/t15774;
                double t15776 = t15775*t15769*(1.0/2.0);
                double t15777 = t15776+1.0;
                double t15778 = log(t15777);
                double t15779 = a1a*c*t15718;
                double t15780 = t15779+1.0;
                double t15781 = t15743*t15744;
                double t15782 = t15781-1.0;
                double t15783 = Aa*t15742*t15780*t15782*t15747*t15768*t15778*2.0;
                double t15793 = t15742*t15743*t15744*t15747*t15767;
                double t15784 = t15736+t15783-t15793;
                double t15785 = t15711*t15754*t15784;
                double t15786 = exp(t15785);
                double t15787 = t15786-1.0;
                double t15788 = 1.0/t15787;
                double t15789 = gamma_ab*2.0;
                double t15790 = gamma_aa+gamma_bb+t15789;
                double t15791 = bet*t15711*t15712*t15713*t15714*t15753*t15790*t15788*(1.0/1.6E1);
                double t15794 = 1.0/(t15710*t15710);
                double t15814 = t15716*t15794;
                double t15795 = t15715-t15814;
                double t15796 = 1.0/pow(t15710,4.0/3.0);
                double t15797 = t15791+1.0;
                double t15798 = bet*bet;
                double t15799 = 1.0/(gammas*gammas);
                double t15800 = 1.0/(k*k);
                double t15801 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t15802 = 1.0/pow(t15710,1.4E1/3.0);
                double t15803 = 1.0/(t15787*t15787);
                double t15804 = 1.0/(t15752*t15752*t15752*t15752);
                double t15805 = t15790*t15790;
                double t15806 = t15800*t15801*t15802*t15803*t15804*t15805*t15798*t15799*(1.0/2.56E2);
                double t15807 = t15806+t15791+1.0;
                double t15808 = 1.0/t15807;
                double t15809 = bet*t15711*t15712*t15713*t15714*t15753*t15790*t15808*t15797*(1.0/1.6E1);
                double t15810 = t15809+1.0;
                double t15811 = t15752*t15752;
                double t15812 = 1.0/pow(t15710,1.0E1/3.0);
                double t15813 = 1.0/pow(t15738,1.0/3.0);
                double t15815 = t15813*t15795*(1.0/3.0);
                double t15816 = 1.0/pow(t15739,1.0/3.0);
                double t15833 = t15816*t15795*(1.0/3.0);
                double t15817 = t15815-t15833;
                double t15818 = 1.0/t15732;
                double t15819 = 1.0/(t15729*t15729);
                double t15820 = 1.0/pow(t15710,5.0/3.0);
                double t15821 = b4p*t15820*t15722*(2.0/3.0);
                double t15822 = b2p*c*t15796*(1.0/3.0);
                double t15823 = 1.0/sqrt(t15719);
                double t15824 = b1p*c*t15823*t15796*(1.0/6.0);
                double t15825 = b3p*c*t15720*t15796*(1.0/2.0);
                double t15826 = t15821+t15822+t15824+t15825;
                double t15827 = t15735*t15826*t15818*t15819;
                double t15828 = 1.0/(t15710*t15710*t15710*t15710*t15710);
                double t15829 = pow(t15738,1.0/3.0);
                double t15830 = t15829*t15795*(4.0/3.0);
                double t15831 = pow(t15739,1.0/3.0);
                double t15837 = t15831*t15795*(4.0/3.0);
                double t15832 = t15830-t15837;
                double t15834 = bet*t15711*t15712*t15713*t15812*t15753*t15790*t15788*(7.0/4.8E1);
                double t15835 = bet*t15711*t15712*t15713*t15714*t15754*t15790*t15817*t15788*(1.0/8.0);
                double t15836 = t15742*t15716*t15743*t15737*t15747*t15767*4.0;
                double t15838 = t15742*t15832*t15743*t15744*t15767;
                double t15839 = 1.0/t15763;
                double t15840 = 1.0/(t15760*t15760);
                double t15841 = b4f*t15820*t15722*(2.0/3.0);
                double t15842 = b2f*c*t15796*(1.0/3.0);
                double t15843 = b1f*c*t15823*t15796*(1.0/6.0);
                double t15844 = b3f*c*t15720*t15796*(1.0/2.0);
                double t15845 = t15841+t15842+t15843+t15844;
                double t15846 = a1f*c*c0f*t15764*t15796*(2.0/3.0);
                double t15847 = a1p*c*c0p*t15733*t15796*(2.0/3.0);
                double t15848 = t15744*t15828*4.0;
                double t15862 = t15716*t15743*t15737*4.0;
                double t15849 = -t15862+t15848;
                double t15850 = Aa*t15742*t15780*t15747*t15768*t15849*t15778*2.0;
                double t15851 = 1.0/t15777;
                double t15852 = 1.0/(t15774*t15774);
                double t15853 = b4a*t15820*t15722*(2.0/3.0);
                double t15854 = b2a*c*t15796*(1.0/3.0);
                double t15855 = b1a*c*t15823*t15796*(1.0/6.0);
                double t15856 = b3a*c*t15720*t15796*(1.0/2.0);
                double t15857 = t15853+t15854+t15855+t15856;
                double t15858 = Aa*a1a*c*t15742*t15782*t15747*t15768*t15778*t15796*(2.0/3.0);
                double t15859 = t15711*t15804*t15817*t15784*3.0;
                double t15863 = t15840*t15845*t15766*t15839;
                double t15860 = t15827-t15863+t15846-t15847;
                double t15861 = t15742*t15743*t15860*t15744*t15747;
                double t15864 = t15862-t15848;
                double t15865 = log(t15810);
                v_rho_a[Q] += scale * (-t15736-t15783+t15710*(t15861-t15827+t15836+t15838+t15847+t15858+gammas*t15811*t15817*t15865*3.0-(gammas*t15811*t15752*(bet*t15711*t15712*t15713*t15714*t15753*t15790*t15808*(t15834+t15835-bet*t15711*t15712*t15713*t15803*t15714*t15753*t15790*t15786*(t15859+t15711*t15754*(t15850-t15827+t15836+t15838+t15847+t15858+t15742*t15743*t15744*t15747*(t15827+t15846-t15840*t15845*t15766*t15839-a1p*c*c0p*t15733*t15796*(2.0/3.0))-t15742*t15744*t15747*t15828*t15767*4.0-Aa*t15742*t15832*t15780*t15782*t15768*t15778*2.0-t15742*t15851*t15780*t15852*t15782*t15747*t15857*t15768))*(1.0/1.6E1))*(1.0/1.6E1)+bet*t15711*t15712*t15713*t15812*t15753*t15790*t15808*t15797*(7.0/4.8E1)-bet*t15711*t15712*t15713*t15714*t15753*1.0/(t15807*t15807)*t15790*t15797*(t15834+t15835+1.0/pow(t15710,1.7E1/3.0)*t15800*t15801*t15803*t15804*t15805*t15798*t15799*(7.0/3.84E2)+t15800*t15801*t15802*t15803*t15805*1.0/(t15752*t15752*t15752*t15752*t15752)*t15817*t15798*t15799*(1.0/6.4E1)-bet*t15711*t15712*t15713*t15803*t15714*t15753*t15790*t15786*(t15859+t15711*t15754*(t15861-t15827+t15836+t15838+t15847+t15858-t15742*t15744*t15747*t15828*t15767*4.0-Aa*t15742*t15832*t15780*t15782*t15768*t15778*2.0-Aa*t15742*t15780*t15747*t15864*t15768*t15778*2.0-t15742*t15851*t15780*t15852*t15782*t15747*t15857*t15768))*(1.0/1.6E1)-t15800*t15801*t15802*t15804*t15805*t15786*1.0/(t15787*t15787*t15787)*t15798*t15799*(t15859+t15711*t15754*(t15850+t15861-t15827+t15836+t15838+t15847+t15858-t15742*t15744*t15747*t15828*t15767*4.0-Aa*t15742*t15832*t15780*t15782*t15768*t15778*2.0-t15742*t15851*t15780*t15852*t15782*t15747*t15857*t15768))*(1.0/1.28E2))*(1.0/1.6E1)+bet*t15711*t15712*t15713*t15714*t15754*t15790*t15808*t15817*t15797*(1.0/8.0)))/t15810-t15742*t15744*t15747*t15828*(t15736-t15792)*4.0-Aa*t15742*t15832*t15780*t15782*t15768*t15778*2.0-Aa*t15742*t15780*t15747*t15864*t15768*t15778*2.0-t15742*t15851*t15780*t15852*t15782*t15747*t15857*t15768)+gammas*t15811*t15752*t15865+t15742*t15743*t15744*t15747*(t15736-t15792));
            }

            // v_rho_b
            if (deriv >= 1) {
                double t15867 = rho_a+rho_b;
                double t15868 = 1.0/gammas;
                double t15869 = 1.0/k;
                double t15870 = 1.0/(pi_m12*pi_m12);
                double t15871 = 1.0/pow(t15867,7.0/3.0);
                double t15872 = 1.0/t15867;
                double t15873 = rho_a-rho_b;
                double t15874 = t15872*t15873;
                double t15875 = 1.0/pow(t15867,1.0/3.0);
                double t15876 = c*t15875;
                double t15877 = sqrt(t15876);
                double t15878 = pow(t15876,3.0/2.0);
                double t15879 = c*c;
                double t15880 = 1.0/pow(t15867,2.0/3.0);
                double t15881 = 1.0/c0p;
                double t15882 = b1p*t15877;
                double t15883 = b3p*t15878;
                double t15884 = b4p*t15880*t15879;
                double t15885 = b2p*c*t15875;
                double t15886 = t15882+t15883+t15884+t15885;
                double t15887 = 1.0/t15886;
                double t15888 = t15881*t15887*(1.0/2.0);
                double t15889 = t15888+1.0;
                double t15890 = log(t15889);
                double t15891 = a1p*c*t15875;
                double t15892 = t15891+1.0;
                double t15893 = c0p*t15890*t15892*2.0;
                double t15894 = t15873*t15873;
                double t15895 = t15874+1.0;
                double t15896 = -t15874+1.0;
                double t15897 = two_13*2.0;
                double t15898 = t15897-2.0;
                double t15899 = 1.0/t15898;
                double t15900 = 1.0/(t15867*t15867*t15867*t15867);
                double t15901 = t15894*t15894;
                double t15902 = pow(t15895,4.0/3.0);
                double t15903 = pow(t15896,4.0/3.0);
                double t15904 = t15902+t15903-2.0;
                double t15905 = pow(t15895,2.0/3.0);
                double t15906 = t15905*(1.0/2.0);
                double t15907 = pow(t15896,2.0/3.0);
                double t15908 = t15907*(1.0/2.0);
                double t15909 = t15906+t15908;
                double t15910 = 1.0/(t15909*t15909);
                double t15911 = 1.0/(t15909*t15909*t15909);
                double t15912 = 1.0/c0f;
                double t15913 = b1f*t15877;
                double t15914 = b3f*t15878;
                double t15915 = b4f*t15880*t15879;
                double t15916 = b2f*c*t15875;
                double t15917 = t15913+t15914+t15915+t15916;
                double t15918 = 1.0/t15917;
                double t15919 = t15912*t15918*(1.0/2.0);
                double t15920 = t15919+1.0;
                double t15921 = log(t15920);
                double t15922 = a1f*c*t15875;
                double t15923 = t15922+1.0;
                double t15949 = c0f*t15921*t15923*2.0;
                double t15924 = t15893-t15949;
                double t15925 = 1.0/d2fz0;
                double t15926 = 1.0/Aa;
                double t15927 = b1a*t15877;
                double t15928 = b3a*t15878;
                double t15929 = b4a*t15880*t15879;
                double t15930 = b2a*c*t15875;
                double t15931 = t15930+t15927+t15928+t15929;
                double t15932 = 1.0/t15931;
                double t15933 = t15932*t15926*(1.0/2.0);
                double t15934 = t15933+1.0;
                double t15935 = log(t15934);
                double t15936 = a1a*c*t15875;
                double t15937 = t15936+1.0;
                double t15938 = t15900*t15901;
                double t15939 = t15938-1.0;
                double t15940 = Aa*t15904*t15925*t15935*t15937*t15939*t15899*2.0;
                double t15950 = t15900*t15901*t15904*t15924*t15899;
                double t15941 = t15940-t15950+t15893;
                double t15942 = t15911*t15941*t15868;
                double t15943 = exp(t15942);
                double t15944 = t15943-1.0;
                double t15945 = 1.0/t15944;
                double t15946 = gamma_ab*2.0;
                double t15947 = gamma_aa+gamma_bb+t15946;
                double t15948 = bet*t15910*t15870*t15871*t15945*t15947*t15868*t15869*(1.0/1.6E1);
                double t15951 = 1.0/(t15867*t15867);
                double t15952 = t15951*t15873;
                double t15953 = t15952+t15872;
                double t15954 = 1.0/pow(t15867,4.0/3.0);
                double t15955 = t15948+1.0;
                double t15956 = bet*bet;
                double t15957 = 1.0/(gammas*gammas);
                double t15958 = 1.0/(k*k);
                double t15959 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t15960 = 1.0/pow(t15867,1.4E1/3.0);
                double t15961 = 1.0/(t15944*t15944);
                double t15962 = 1.0/(t15909*t15909*t15909*t15909);
                double t15963 = t15947*t15947;
                double t15964 = t15960*t15961*t15962*t15963*t15956*t15957*t15958*t15959*(1.0/2.56E2);
                double t15965 = t15964+t15948+1.0;
                double t15966 = 1.0/t15965;
                double t15967 = bet*t15910*t15870*t15871*t15955*t15947*t15966*t15868*t15869*(1.0/1.6E1);
                double t15968 = t15967+1.0;
                double t15969 = t15909*t15909;
                double t15970 = 1.0/pow(t15867,1.0E1/3.0);
                double t15971 = 1.0/pow(t15895,1.0/3.0);
                double t15972 = t15953*t15971*(1.0/3.0);
                double t15973 = 1.0/pow(t15896,1.0/3.0);
                double t15990 = t15953*t15973*(1.0/3.0);
                double t15974 = t15972-t15990;
                double t15975 = 1.0/t15889;
                double t15976 = 1.0/(t15886*t15886);
                double t15977 = 1.0/pow(t15867,5.0/3.0);
                double t15978 = b4p*t15977*t15879*(2.0/3.0);
                double t15979 = b2p*c*t15954*(1.0/3.0);
                double t15980 = 1.0/sqrt(t15876);
                double t15981 = b1p*c*t15980*t15954*(1.0/6.0);
                double t15982 = b3p*c*t15954*t15877*(1.0/2.0);
                double t15983 = t15981+t15982+t15978+t15979;
                double t15984 = 1.0/(t15867*t15867*t15867*t15867*t15867);
                double t15985 = pow(t15895,1.0/3.0);
                double t15986 = t15953*t15985*(4.0/3.0);
                double t15987 = pow(t15896,1.0/3.0);
                double t15994 = t15953*t15987*(4.0/3.0);
                double t15988 = -t15994+t15986;
                double t15989 = t15892*t15983*t15975*t15976;
                double t15991 = bet*t15910*t15870*t15970*t15945*t15947*t15868*t15869*(7.0/4.8E1);
                double t15992 = t15900*t15904*t15924*t15873*t15894*t15899*4.0;
                double t15993 = t15901*t15904*t15924*t15984*t15899*4.0;
                double t15995 = t15900*t15901*t15924*t15988*t15899;
                double t15996 = 1.0/t15920;
                double t15997 = 1.0/(t15917*t15917);
                double t15998 = b4f*t15977*t15879*(2.0/3.0);
                double t15999 = b2f*c*t15954*(1.0/3.0);
                double t16000 = b1f*c*t15980*t15954*(1.0/6.0);
                double t16001 = b3f*c*t15954*t15877*(1.0/2.0);
                double t16002 = t15998+t15999+t16000+t16001;
                double t16003 = a1f*c*c0f*t15921*t15954*(2.0/3.0);
                double t16004 = t15900*t15873*t15894*4.0;
                double t16005 = t15901*t15984*4.0;
                double t16006 = t16004+t16005;
                double t16007 = 1.0/t15934;
                double t16008 = 1.0/(t15931*t15931);
                double t16009 = b4a*t15977*t15879*(2.0/3.0);
                double t16010 = b2a*c*t15954*(1.0/3.0);
                double t16011 = b1a*c*t15980*t15954*(1.0/6.0);
                double t16012 = b3a*c*t15954*t15877*(1.0/2.0);
                double t16013 = t16010+t16011+t16012+t16009;
                double t16014 = t15904*t15925*t15937*t15939*t15899*t16013*t16007*t16008;
                double t16016 = a1p*c*c0p*t15890*t15954*(2.0/3.0);
                double t16018 = t15923*t15996*t15997*t16002;
                double t16015 = t15989+t16003-t16016-t16018;
                double t16017 = t15941*t15962*t15974*t15868*3.0;
                double t16022 = t15900*t15901*t15904*t15899*t16015;
                double t16023 = Aa*t15925*t15935*t15937*t15939*t15988*t15899*2.0;
                double t16024 = Aa*t15904*t15925*t15935*t15937*t15899*t16006*2.0;
                double t16025 = Aa*a1a*c*t15904*t15925*t15935*t15954*t15939*t15899*(2.0/3.0);
                double t16019 = t15992+t15993+t15995+t15989-t16022+t16014-t16023-t16024-t16016-t16025;
                double t16020 = t15911*t15868*t16019;
                double t16021 = t16020+t16017;
                double t16026 = bet*t15910*t15870*t15871*t15943*t15961*t15947*t15868*t15869*t16021*(1.0/1.6E1);
                double t16027 = log(t15968);
                v_rho_b[Q] += scale * (-t15940-t15893-t15867*(t15992+t15993+t15995+t15989-t16022+t16014-t16023-t16024-t16016-t16025+gammas*t15974*t15969*t16027*3.0+(gammas*t15909*t15969*(bet*t15910*t15870*t15970*t15955*t15947*t15966*t15868*t15869*(7.0/4.8E1)+bet*t15910*t15870*t15871*t15947*t15966*t15868*t15869*(t15991+t16026-bet*t15911*t15870*t15871*t15945*t15947*t15974*t15868*t15869*(1.0/8.0))*(1.0/1.6E1)-bet*t15911*t15870*t15871*t15955*t15947*t15974*t15966*t15868*t15869*(1.0/8.0)-bet*t15910*t15870*t15871*t15955*t15947*1.0/(t15965*t15965)*t15868*t15869*(t15991+t16026+t15961*t15962*t15963*t15956*1.0/pow(t15867,1.7E1/3.0)*t15957*t15958*t15959*(7.0/3.84E2)-bet*t15911*t15870*t15871*t15945*t15947*t15974*t15868*t15869*(1.0/8.0)-t15960*t15961*1.0/(t15909*t15909*t15909*t15909*t15909)*t15963*t15956*t15974*t15957*t15958*t15959*(1.0/6.4E1)+t15960*t15943*1.0/(t15944*t15944*t15944)*t15962*t15963*t15956*t15957*t15958*t15959*t16021*(1.0/1.28E2))*(1.0/1.6E1)))/t15968)+gammas*t15909*t15969*t16027+t15900*t15901*t15904*t15899*(t15893-t15949));
            }

            // v_gamma_aa
            if (deriv >= 1) {
                double t16029 = rho_a+rho_b;
                double t16030 = 1.0/gammas;
                double t16031 = 1.0/k;
                double t16032 = 1.0/(pi_m12*pi_m12);
                double t16033 = 1.0/pow(t16029,7.0/3.0);
                double t16034 = 1.0/t16029;
                double t16035 = rho_a-rho_b;
                double t16036 = t16034*t16035;
                double t16037 = 1.0/pow(t16029,1.0/3.0);
                double t16038 = c*t16037;
                double t16039 = sqrt(t16038);
                double t16040 = pow(t16038,3.0/2.0);
                double t16041 = c*c;
                double t16042 = 1.0/pow(t16029,2.0/3.0);
                double t16043 = 1.0/c0p;
                double t16044 = b1p*t16039;
                double t16045 = b3p*t16040;
                double t16046 = b4p*t16041*t16042;
                double t16047 = b2p*c*t16037;
                double t16048 = t16044+t16045+t16046+t16047;
                double t16049 = 1.0/t16048;
                double t16050 = t16043*t16049*(1.0/2.0);
                double t16051 = t16050+1.0;
                double t16052 = log(t16051);
                double t16053 = a1p*c*t16037;
                double t16054 = t16053+1.0;
                double t16055 = c0p*t16052*t16054*2.0;
                double t16056 = t16035*t16035;
                double t16057 = t16036+1.0;
                double t16058 = -t16036+1.0;
                double t16059 = two_13*2.0;
                double t16060 = t16059-2.0;
                double t16061 = 1.0/t16060;
                double t16062 = 1.0/(t16029*t16029*t16029*t16029);
                double t16063 = t16056*t16056;
                double t16064 = pow(t16057,4.0/3.0);
                double t16065 = pow(t16058,4.0/3.0);
                double t16066 = t16064+t16065-2.0;
                double t16067 = pow(t16057,2.0/3.0);
                double t16068 = t16067*(1.0/2.0);
                double t16069 = pow(t16058,2.0/3.0);
                double t16070 = t16069*(1.0/2.0);
                double t16071 = t16070+t16068;
                double t16072 = 1.0/(t16071*t16071);
                double t16073 = 1.0/(t16071*t16071*t16071);
                double t16074 = 1.0/c0f;
                double t16075 = b1f*t16039;
                double t16076 = b3f*t16040;
                double t16077 = b4f*t16041*t16042;
                double t16078 = b2f*c*t16037;
                double t16079 = t16075+t16076+t16077+t16078;
                double t16080 = 1.0/t16079;
                double t16081 = t16080*t16074*(1.0/2.0);
                double t16082 = t16081+1.0;
                double t16083 = log(t16082);
                double t16084 = a1f*c*t16037;
                double t16085 = t16084+1.0;
                double t16111 = c0f*t16083*t16085*2.0;
                double t16086 = -t16111+t16055;
                double t16087 = 1.0/d2fz0;
                double t16088 = 1.0/Aa;
                double t16089 = b1a*t16039;
                double t16090 = b3a*t16040;
                double t16091 = b4a*t16041*t16042;
                double t16092 = b2a*c*t16037;
                double t16093 = t16090+t16091+t16092+t16089;
                double t16094 = 1.0/t16093;
                double t16095 = t16094*t16088*(1.0/2.0);
                double t16096 = t16095+1.0;
                double t16097 = log(t16096);
                double t16098 = a1a*c*t16037;
                double t16099 = t16098+1.0;
                double t16100 = t16062*t16063;
                double t16101 = t16100-1.0;
                double t16102 = Aa*t16101*t16061*t16066*t16087*t16097*t16099*2.0;
                double t16103 = t16102+t16055-t16061*t16062*t16063*t16066*t16086;
                double t16104 = t16030*t16103*t16073;
                double t16105 = exp(t16104);
                double t16106 = t16105-1.0;
                double t16107 = 1.0/t16106;
                double t16108 = gamma_ab*2.0;
                double t16109 = gamma_aa+gamma_bb+t16108;
                double t16110 = bet*t16030*t16031*t16032*t16033*t16107*t16072*t16109*(1.0/1.6E1);
                double t16112 = t16071*t16071;
                double t16113 = t16111-t16055;
                double t16114 = t16113*t16061*t16062*t16063*t16066;
                double t16115 = t16102+t16114+t16055;
                double t16116 = t16030*t16115*t16073;
                double t16117 = exp(t16116);
                double t16118 = t16117-1.0;
                double t16119 = 1.0/t16118;
                double t16120 = bet*t16030*t16031*t16032*t16033*t16072*t16109*t16119*(1.0/1.6E1);
                double t16121 = bet*bet;
                double t16122 = 1.0/(gammas*gammas);
                double t16123 = 1.0/(k*k);
                double t16124 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16125 = 1.0/pow(t16029,1.4E1/3.0);
                double t16126 = 1.0/(t16118*t16118);
                double t16127 = 1.0/(t16071*t16071*t16071*t16071);
                double t16128 = t16109*t16109;
                double t16129 = t16121*t16122*t16123*t16124*t16125*t16126*t16127*t16128*(1.0/2.56E2);
                double t16130 = t16120+t16129+1.0;
                double t16131 = 1.0/t16130;
                double t16132 = t16120+1.0;
                v_gamma_aa[Q] += scale * ((gammas*t16112*t16071*t16029*(bet*t16030*t16031*t16032*t16131*t16033*t16132*t16072*(1.0/1.6E1)+t16121*t16122*t16131*t16123*t16124*t16125*t16109*t16127*t16119*(1.0/2.56E2)-bet*t16030*t16031*1.0/(t16130*t16130)*t16032*t16033*t16132*t16072*t16109*(bet*t16030*t16031*t16032*t16033*t16072*t16119*(1.0/1.6E1)+t16121*t16122*t16123*t16124*t16125*t16126*t16127*(gamma_aa*2.0+gamma_ab*4.0+gamma_bb*2.0)*(1.0/2.56E2))*(1.0/1.6E1)))/((bet*t16030*t16031*t16032*t16033*t16072*t16109*(t16110+1.0)*(1.0/1.6E1))/(t16110+t16129+1.0)+1.0));
            }

            // v_gamma_ab
            if (deriv >= 1) {
                double t16134 = rho_a+rho_b;
                double t16135 = 1.0/gammas;
                double t16136 = 1.0/k;
                double t16137 = 1.0/(pi_m12*pi_m12);
                double t16138 = 1.0/pow(t16134,7.0/3.0);
                double t16139 = 1.0/t16134;
                double t16140 = rho_a-rho_b;
                double t16141 = t16140*t16139;
                double t16142 = 1.0/pow(t16134,1.0/3.0);
                double t16143 = c*t16142;
                double t16144 = sqrt(t16143);
                double t16145 = pow(t16143,3.0/2.0);
                double t16146 = c*c;
                double t16147 = 1.0/pow(t16134,2.0/3.0);
                double t16148 = 1.0/c0p;
                double t16149 = b1p*t16144;
                double t16150 = b3p*t16145;
                double t16151 = b4p*t16146*t16147;
                double t16152 = b2p*c*t16142;
                double t16153 = t16150+t16151+t16152+t16149;
                double t16154 = 1.0/t16153;
                double t16155 = t16154*t16148*(1.0/2.0);
                double t16156 = t16155+1.0;
                double t16157 = log(t16156);
                double t16158 = a1p*c*t16142;
                double t16159 = t16158+1.0;
                double t16160 = c0p*t16157*t16159*2.0;
                double t16161 = t16140*t16140;
                double t16162 = t16141+1.0;
                double t16163 = -t16141+1.0;
                double t16164 = two_13*2.0;
                double t16165 = t16164-2.0;
                double t16166 = 1.0/t16165;
                double t16167 = 1.0/(t16134*t16134*t16134*t16134);
                double t16168 = t16161*t16161;
                double t16169 = pow(t16162,4.0/3.0);
                double t16170 = pow(t16163,4.0/3.0);
                double t16171 = t16170+t16169-2.0;
                double t16172 = pow(t16162,2.0/3.0);
                double t16173 = t16172*(1.0/2.0);
                double t16174 = pow(t16163,2.0/3.0);
                double t16175 = t16174*(1.0/2.0);
                double t16176 = t16173+t16175;
                double t16177 = 1.0/(t16176*t16176);
                double t16178 = 1.0/(t16176*t16176*t16176);
                double t16179 = 1.0/c0f;
                double t16180 = b1f*t16144;
                double t16181 = b3f*t16145;
                double t16182 = b4f*t16146*t16147;
                double t16183 = b2f*c*t16142;
                double t16184 = t16180+t16181+t16182+t16183;
                double t16185 = 1.0/t16184;
                double t16186 = t16185*t16179*(1.0/2.0);
                double t16187 = t16186+1.0;
                double t16188 = log(t16187);
                double t16189 = a1f*c*t16142;
                double t16190 = t16189+1.0;
                double t16216 = c0f*t16190*t16188*2.0;
                double t16191 = t16160-t16216;
                double t16192 = 1.0/d2fz0;
                double t16193 = 1.0/Aa;
                double t16194 = b1a*t16144;
                double t16195 = b3a*t16145;
                double t16196 = b4a*t16146*t16147;
                double t16197 = b2a*c*t16142;
                double t16198 = t16194+t16195+t16196+t16197;
                double t16199 = 1.0/t16198;
                double t16200 = t16193*t16199*(1.0/2.0);
                double t16201 = t16200+1.0;
                double t16202 = log(t16201);
                double t16203 = a1a*c*t16142;
                double t16204 = t16203+1.0;
                double t16205 = t16167*t16168;
                double t16206 = t16205-1.0;
                double t16207 = Aa*t16202*t16204*t16206*t16171*t16192*t16166*2.0;
                double t16217 = t16171*t16191*t16166*t16167*t16168;
                double t16208 = t16160+t16207-t16217;
                double t16209 = t16135*t16208*t16178;
                double t16210 = exp(t16209);
                double t16211 = t16210-1.0;
                double t16212 = 1.0/t16211;
                double t16213 = gamma_ab*2.0;
                double t16214 = gamma_aa+gamma_bb+t16213;
                double t16215 = bet*t16212*t16214*t16135*t16136*t16137*t16138*t16177*(1.0/1.6E1);
                double t16218 = t16176*t16176;
                double t16219 = t16215+1.0;
                double t16220 = bet*bet;
                double t16221 = 1.0/(gammas*gammas);
                double t16222 = 1.0/(k*k);
                double t16223 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16224 = 1.0/pow(t16134,1.4E1/3.0);
                double t16225 = 1.0/(t16211*t16211);
                double t16226 = 1.0/(t16176*t16176*t16176*t16176);
                double t16227 = t16214*t16214;
                double t16228 = t16220*t16221*t16222*t16223*t16224*t16225*t16226*t16227*(1.0/2.56E2);
                double t16229 = t16215+t16228+1.0;
                double t16230 = 1.0/t16229;
                v_gamma_ab[Q] += scale * ((gammas*t16134*t16218*t16176*(bet*t16230*t16135*t16136*t16137*t16138*t16219*t16177*(1.0/8.0)+t16220*t16212*t16221*t16230*t16222*t16214*t16223*t16224*t16226*(1.0/1.28E2)-bet*t16214*t16135*t16136*t16137*t16138*t16219*1.0/(t16229*t16229)*t16177*(bet*t16212*t16135*t16136*t16137*t16138*t16177*(1.0/8.0)+t16220*t16221*t16222*t16223*t16224*t16225*t16226*(gamma_aa*4.0+gamma_ab*8.0+gamma_bb*4.0)*(1.0/2.56E2))*(1.0/1.6E1)))/(bet*t16230*t16214*t16135*t16136*t16137*t16138*t16219*t16177*(1.0/1.6E1)+1.0));
            }

            // v_gamma_bb
            if (deriv >= 1) {
                double t16232 = rho_a+rho_b;
                double t16233 = 1.0/gammas;
                double t16234 = 1.0/k;
                double t16235 = 1.0/(pi_m12*pi_m12);
                double t16236 = 1.0/pow(t16232,7.0/3.0);
                double t16237 = 1.0/t16232;
                double t16238 = rho_a-rho_b;
                double t16239 = t16237*t16238;
                double t16240 = 1.0/pow(t16232,1.0/3.0);
                double t16241 = c*t16240;
                double t16242 = sqrt(t16241);
                double t16243 = pow(t16241,3.0/2.0);
                double t16244 = c*c;
                double t16245 = 1.0/pow(t16232,2.0/3.0);
                double t16246 = 1.0/c0p;
                double t16247 = b1p*t16242;
                double t16248 = b3p*t16243;
                double t16249 = b4p*t16244*t16245;
                double t16250 = b2p*c*t16240;
                double t16251 = t16250+t16247+t16248+t16249;
                double t16252 = 1.0/t16251;
                double t16253 = t16252*t16246*(1.0/2.0);
                double t16254 = t16253+1.0;
                double t16255 = log(t16254);
                double t16256 = a1p*c*t16240;
                double t16257 = t16256+1.0;
                double t16258 = c0p*t16255*t16257*2.0;
                double t16259 = t16238*t16238;
                double t16260 = t16239+1.0;
                double t16261 = -t16239+1.0;
                double t16262 = two_13*2.0;
                double t16263 = t16262-2.0;
                double t16264 = 1.0/t16263;
                double t16265 = 1.0/(t16232*t16232*t16232*t16232);
                double t16266 = t16259*t16259;
                double t16267 = pow(t16260,4.0/3.0);
                double t16268 = pow(t16261,4.0/3.0);
                double t16269 = t16267+t16268-2.0;
                double t16270 = pow(t16260,2.0/3.0);
                double t16271 = t16270*(1.0/2.0);
                double t16272 = pow(t16261,2.0/3.0);
                double t16273 = t16272*(1.0/2.0);
                double t16274 = t16271+t16273;
                double t16275 = 1.0/(t16274*t16274);
                double t16276 = 1.0/(t16274*t16274*t16274);
                double t16277 = 1.0/c0f;
                double t16278 = b1f*t16242;
                double t16279 = b3f*t16243;
                double t16280 = b4f*t16244*t16245;
                double t16281 = b2f*c*t16240;
                double t16282 = t16280+t16281+t16278+t16279;
                double t16283 = 1.0/t16282;
                double t16284 = t16283*t16277*(1.0/2.0);
                double t16285 = t16284+1.0;
                double t16286 = log(t16285);
                double t16287 = a1f*c*t16240;
                double t16288 = t16287+1.0;
                double t16314 = c0f*t16286*t16288*2.0;
                double t16289 = -t16314+t16258;
                double t16290 = 1.0/d2fz0;
                double t16291 = 1.0/Aa;
                double t16292 = b1a*t16242;
                double t16293 = b3a*t16243;
                double t16294 = b4a*t16244*t16245;
                double t16295 = b2a*c*t16240;
                double t16296 = t16292+t16293+t16294+t16295;
                double t16297 = 1.0/t16296;
                double t16298 = t16291*t16297*(1.0/2.0);
                double t16299 = t16298+1.0;
                double t16300 = log(t16299);
                double t16301 = a1a*c*t16240;
                double t16302 = t16301+1.0;
                double t16303 = t16265*t16266;
                double t16304 = t16303-1.0;
                double t16305 = Aa*t16300*t16302*t16304*t16290*t16264*t16269*2.0;
                double t16306 = t16305+t16258-t16264*t16265*t16266*t16269*t16289;
                double t16307 = t16233*t16306*t16276;
                double t16308 = exp(t16307);
                double t16309 = t16308-1.0;
                double t16310 = 1.0/t16309;
                double t16311 = gamma_ab*2.0;
                double t16312 = gamma_aa+gamma_bb+t16311;
                double t16313 = bet*t16310*t16312*t16233*t16234*t16235*t16236*t16275*(1.0/1.6E1);
                double t16315 = t16274*t16274;
                double t16316 = t16314-t16258;
                double t16317 = t16316*t16264*t16265*t16266*t16269;
                double t16318 = t16305+t16317+t16258;
                double t16319 = t16233*t16318*t16276;
                double t16320 = exp(t16319);
                double t16321 = t16320-1.0;
                double t16322 = 1.0/t16321;
                double t16323 = bet*t16312*t16322*t16233*t16234*t16235*t16236*t16275*(1.0/1.6E1);
                double t16324 = bet*bet;
                double t16325 = 1.0/(gammas*gammas);
                double t16326 = 1.0/(k*k);
                double t16327 = 1.0/(pi_m12*pi_m12*pi_m12*pi_m12);
                double t16328 = 1.0/pow(t16232,1.4E1/3.0);
                double t16329 = 1.0/(t16321*t16321);
                double t16330 = 1.0/(t16274*t16274*t16274*t16274);
                double t16331 = t16312*t16312;
                double t16332 = t16330*t16331*t16324*t16325*t16326*t16327*t16328*t16329*(1.0/2.56E2);
                double t16333 = t16323+t16332+1.0;
                double t16334 = 1.0/t16333;
                double t16335 = t16323+1.0;
                v_gamma_bb[Q] += scale * ((gammas*t16232*t16315*t16274*(bet*t16233*t16234*t16235*t16334*t16236*t16335*t16275*(1.0/1.6E1)+t16312*t16330*t16322*t16324*t16325*t16334*t16326*t16327*t16328*(1.0/2.56E2)-bet*t16312*t16233*t16234*1.0/(t16333*t16333)*t16235*t16236*t16335*t16275*(bet*t16322*t16233*t16234*t16235*t16236*t16275*(1.0/1.6E1)+t16330*t16324*t16325*t16326*t16327*t16328*t16329*(gamma_aa*2.0+gamma_ab*4.0+gamma_bb*2.0)*(1.0/2.56E2))*(1.0/1.6E1)))/((bet*t16312*t16233*t16234*t16235*t16236*t16275*(t16313+1.0)*(1.0/1.6E1))/(t16313+t16332+1.0)+1.0));
            }

        }
    }
}

}
