% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/binomial.regression.R
\name{binreg}
\alias{binreg}
\alias{logitIPCW}
\title{Binomial Regression for censored competing risks data}
\usage{
binreg(
  formula,
  data,
  cause = 1,
  time = NULL,
  beta = NULL,
  offset = NULL,
  weights = NULL,
  cens.weights = NULL,
  cens.model = ~+1,
  se = TRUE,
  kaplan.meier = TRUE,
  cens.code = 0,
  no.opt = FALSE,
  method = "nr",
  augmentation = NULL,
  ...
)
}
\arguments{
\item{formula}{formula with outcome (see \code{coxph})}

\item{data}{data frame}

\item{cause}{cause of interest}

\item{time}{time of interest}

\item{beta}{starting values}

\item{offset}{offsets for partial likelihood}

\item{weights}{for score equations}

\item{cens.weights}{censoring weights}

\item{cens.model}{only stratified cox model without covariates}

\item{se}{to compute se's  based on IPCW}

\item{kaplan.meier}{uses Kaplan-Meier for IPCW in contrast to exp(-Baseline)}

\item{cens.code}{gives censoring code}

\item{no.opt}{to not optimize}

\item{method}{for optimization}

\item{augmentation}{to augment binomial regression}

\item{...}{Additional arguments to lower level funtions}
}
\description{
Simple version of comp.risk function of timereg for just one time-point thus fitting the model 
\deqn{P(T \leq t, \epsilon=1 | X ) = expit( X^T beta) }
}
\details{
Based on binomial regresion IPCW response estimating equation: 
\deqn{ X ( \Delta I(T \leq t, \epsilon=1 )/G_c(T_i-) - expit( X^T beta)) = 0 }
for IPCW adjusted responses. 

logitIPCW instead considers 
\deqn{ X  I(min(T_i,t) < G_i)/G_c(min(T_i ,t)) ( I(T \leq t, \epsilon=1 ) - expit( X^T beta)) = 0 }
a standard logistic regression with weights that adjust for IPCW. 

variance is based on  \deqn{ \sum w_i^2 } also with IPCW adjustment, and naive.var is variance 
under known censoring model. 

Censoring model may depend on strata.
}
\examples{

data(bmt)
# logistic regresion with IPCW binomial regression 
out <- binreg(Event(time,cause)~tcell+platelet,bmt,time=50)
summary(out)
predict(out,data.frame(tcell=c(0,1),platelet=c(1,1)),se=TRUE)

outs <- binreg(Event(time,cause)~tcell+platelet,bmt,time=50,cens.model=~strata(tcell,platelet))
summary(outs)

## glm with IPCW weights 
outl <- logitIPCW(Event(time,cause)~tcell+platelet,bmt,time=50)
summary(outl)

##########################################
### risk-ratio of different causes #######
##########################################
data(bmt)
bmt$id <- 1:nrow(bmt)
bmt$status <- bmt$cause
bmt$strata <- 1
bmtdob <- bmt
bmtdob$strata <-2
bmtdob <- dtransform(bmtdob,status=1,cause==2)
bmtdob <- dtransform(bmtdob,status=2,cause==1)
###
bmtdob <- rbind(bmt,bmtdob)
dtable(bmtdob,cause+status~strata)

cif1 <- cif(Event(time,cause)~+1,bmt,cause=1)
cif2 <- cif(Event(time,cause)~+1,bmt,cause=2)
bplot(cif1)
bplot(cif2,add=TRUE,col=2)

cifs1 <- binreg(Event(time,cause)~tcell+platelet+age,bmt,cause=1,time=50)
cifs2 <- binreg(Event(time,cause)~tcell+platelet+age,bmt,cause=2,time=50)
summary(cifs1)
summary(cifs2)

cifdob <- binreg(Event(time,status)~-1+factor(strata)+
	 tcell*factor(strata)+platelet*factor(strata)+age*factor(strata)
	 +cluster(id),bmtdob,cause=1,time=50,cens.model=~strata(strata)+cluster(id))
summary(cifdob)

riskratio <- function(p) {
  expit  <- function(z) 1/(1+exp(-z)) ## expit
  Z <- rbind(c(1,0,1,1,0,0,0,0), c(0,1,1,1,0,1,1,0))
  lp <- c(Z \%*\% p)
  p <- expit(lp)
  return(p[1]/p[2])
}

estimate(cifdob,f=riskratio)

}
\author{
Thomas Scheike
}
