use layout;
use core:geometry;
use graphics;
use ui;

/**
 * Custom layouts that are useful when creating presentations.
 */


/**
 * Custom root element. Adds a border, but a different size than layout:Border.
 */
class SlideBorder extends layout:Border {
	init(Presentation p) {
		init() {}

		border = p.border;
	}
}


/**
 * A simple layout that handles a caption and content.
 */
class CaptionLayout extends layout:Layout {
	// First component, the caption.
	private Component? caption;

	// Second component, the content.
	private Component? content;

	// Space below the caption.
	Float space;

	// Create.
	init() {
		init() {
			// Default spacing.
			space = 10;
		}
	}

	// Add a component.
	Layout:Info add(Component c) : override {
		unless (caption) {
			this.caption = c;
			return Layout:Info();
		}

		content = c;
		Layout:Info();
	}

	// Compute the minimum size.
	Size minSize() : override {
		Size sz;
		if (caption) {
			sz = caption.minSize;
			sz.h += space;
		}

		if (content) {
			Size s = content.minSize;
			sz.w = max(sz.w, s.w);
			sz.h += s.h;
		}
		sz;
	}

	// Re-compute the layout.
	void layout() : override {
		Rect p = pos;
		if (caption) {
			Size s = caption.minSize();
			caption.pos = Rect(p.p0, Size(p.size.w, s.h));
			p.p0.y += s.h + space;
		}

		if (content) {
			content.pos = p;
		}
	}

	// Traverse.
	void findAll(fn(Component)->void fn) : override {
		super:findAll(fn);
		if (caption)
			caption.findAll(fn);
		if (content)
			content.findAll(fn);
	}
}

/**
 * Put things in columns. The columns will always be the same size, regardless of their content.
 */
class ColumnLayout extends layout:Layout {
	private Component[] components;

	// Space between columns.
	Float padding;

	// Create.
	init() {
		init() { padding = 10; }
	}

	// Add a component.
	Layout:Info add(Component c) : override {
		components << c;
		Layout:Info();
	}

	// Minimum size.
	Size minSize() : override {
		Size m(-padding, 0);
		for (c in components) {
			var s = c.minSize();
			m.w += padding + s.w;
			m.h = max(m.h, s.h);
		}
		m;
	}

	// Re-compute the layout.
	void layout() : override {
		Rect p = pos;
		Float step = (p.size.w + padding) / components.count.int.float;
		Size partSize(step - padding, p.size.h);

		for (i, c in components) {
			Point p0 = p.p0;
			p0.x += step * i.int.float;

			c.pos = Rect(p0, partSize);
		}
	}

	// Traverse.
	void findAll(fn(Component)->void fn) : override {
		super:findAll(fn);
		for (c in components)
			c.findAll(fn);
	}
}

/**
 * A layout for putting each element below the previous one. Does not attempt to fill the remaining
 * space below all elements. A simpler alternative to the grid layout, possibly combined with anchor
 * elements.
 */
class RowLayout extends layout:Layout {
	private Component[] components;
	private Cardinal anchor;

	// Space between rows.
	Float padding;

	// Create.
	init() {
		init() {
			padding = 10;
			anchor = north;
		}
	}

	// Align elements to the left.
	void left() {
		anchor = northWest;
	}

	// Align elements to the right.
	void right() {
		anchor = northEast;
	}

	// Align elements to the center (default).
	void center() {
		anchor = north;
	}

	// Add a component.
	Layout:Info add(Component c) : override {
		components << c;
		Layout:Info();
	}

	// Minimum size.
	Size minSize() : override {
		Size m(0, -padding);
		for (c in components) {
			var s = c.minSize();
			m.h += padding + s.h;
			m.w = max(m.w, s.w);
		}
		m;
	}

	// Re-compute the layout.
	void layout() : override {
		Rect p = pos;
		Float y = p.p0.y;

		for (c in components) {
			var s = c.minSize();
			s.w = min(s.w, p.size.w);
			s.h = min(s.h, p.p1.y - y);

			Rect box(Point(p.p0.x, y), Point(p.p1.x, y + s.h));
			c.pos = anchor.align(box, s);
			y += s.h + padding;
		}
	}

	// Traverse.
	void findAll(fn(Component)->void fn) : override {
		super:findAll(fn);
		for (c in components)
			c.findAll(fn);
	}
}

/**
 * A layout that fits the contained element according to a cardinal direction.
 */
class AnchorLayout extends layout:Layout {
	// Our component.
	private Component? component;

	// Where do we place the component?
	private Cardinal anchor;

	// Shift the origin.
	Float xShift;
	Float yShift;

	// Create.
	init(Cardinal anchor) {
		init() {
			anchor = anchor;
		}
	}

	// Add a component.
	Layout:Info add(Component c) : override {
		component = c;
		Layout:Info();
	}

	// Minimum size.
	Size minSize() : override {
		if (component) {
			component.minSize;
		} else {
			Size();
		}
	}

	// Re-compute the layout.
	void layout() : override {
		if (component) {
			Rect p = pos;
			Point a = anchor.pick(p) + Size(xShift, yShift);
			Size sz = min(component.minSize(), p.size);
			component.pos = anchor.place(a, sz);
		}
	}

	// Traverse.
	void findAll(fn(Component)->void fn) : override {
		super:findAll(fn);
		if (component)
			component.findAll(fn);
	}
}

// Create easily.
AnchorLayout anchor(Cardinal c) {
	AnchorLayout(c);
}

/**
 * Group an entire sub-layout into one element, so that animations can be applied to the entire
 * group rather than individual elements.
 *
 * Extending this class allows modifiers, such as 'WithBackground' below, to be implemented conveniently.
 */
class Group extends layout:Layout {
	// Components managed inside this layout.
	protected Component[] children;

	// The "real" element we show to the presentation system.
	private Elem elem;

	// Add an element.
	Layout:Info add(Component c) : override {
		children << c;
		Layout:Info();
	}

	// Get our minimum size.
	Size minSize() : override {
		Size s;
		for (c in children)
			s = max(s, c.minSize);
		s;
	}

	// Do layout. Override and call 'layout(Rect)' to modify the layout if desired.
	void layout() : override {
		layout(pos);
	}

	// Call from derived classes to add borders etc.
	void layout(Rect pos) {
		for (c in children) {
			c.pos = pos;
		}
		elem.pos = pos;
	}

	// Add an animation.
	void animation(Animation a) {
		elem.animation(a);
	}

	// Find all child elements (we won't propagate them since we handle that ourselves).
	void findAll(fn(Component)->void fn) {
		super:findAll(fn);

		for (c in children) {
			c.findAll(&elem.addChild);
		}

		ElementWrap(elem).findAll(fn);
	}

	/**
	 * Element that is seen by the presentation.
	 */
	class Elem extends Element {
		// Child elements.
		private Element[] children;

		// Minimum size. Not really needed.
		Size minSize() : override { Size(); }

		// Initialize animations.
		void setupAnimations(Presentation p) {
			super:setupAnimations(p);

			for (c in children)
				c.setupAnimations(p);
		}

		// Extract animations.
		Animation[] animations() : override {
			var here = Animation[](super:animations());
			for (c in children)
				here.append(c.animations());
			here;
		}

		// Low-level drawing.
		void draw(Graphics g, Nat step, Duration time) : override {
			if (before(g, step, time)) {
				for (c in children) {
					c.draw(g, step, time);
				}
			}
			after(g, step, time);
		}

		// Regular drawing (not required).
		void draw(Graphics g) : override {}

		// Add a child.
		void addChild(Component e) {
			if (e as ElementWrap)
				children << e.control;
		}
	}
}

/**
 * Layout that draws a background behind an element.
 */
class WithBackground extends Group {
	// The solid-fill component that paints the background.
	private SolidFill background;

	// Margin to add.
	private Size margin;

	// Create.
	init(Color color) {
		init() {
			background = SolidFill(color);
			margin = Size(5, 5);
		}

		add(ElementWrap(background));
	}

	// Set the margin.
	void margin(Float w, Float h) {
		margin = Size(w, h);
	}

	// Set the border.
	void border(Color c) {
		background.border(c);
	}

	// Minimum size.
	Size minSize() : override {
		super:minSize() + margin*2;
	}

	// Re-compute the layout.
	void layout() : override {
		Rect p = pos;
		p.p0 += margin;
		p.p1 -= margin;
		super:layout(p);

		background.pos = pos;
	}
}


/**
 * Layout that adds a caption to another element.
 */
class WithCaption extends Group {
	// Caption text.
	private Heading text;

	// Space between text and the content.
	private Float space;

	// Direction of the text.
	private Cardinal textDir;

	// Create.
	init(Str text, Cardinal where, TextStyle font) {
		init() {
			text = Heading(text, font);
			space = font.space;
			textDir = where;
		}

		add(ElementWrap(this.text));
	}

	// Compute minimum size.
	Size minSize() {
		Size s;
		for (i, c in children)
			if (i > 0)
				s = max(s, c.minSize());

		Size textSz = text.minSize;
		s.w = max(s.w, textSz.w);
		s.h += space + textSz.h;
		s;
	}

	// Re-compute the layout.
	void layout() {
		Rect p = pos;
		Point dir = textDir.direction;

		Size textSize = text.minSize;
		Rect textRect = p;
		Rect contentRect = p;
		if (dir.y > 0) {
			textRect.p0.y = p.p1.y - textSize.h;
			contentRect.p1.y = textRect.p0.y - space;
		} else {
			textRect.p1.y = p.p0.y + textSize.h;
			contentRect.p0.y = textRect.p1.y + space;
		}

		super:layout(contentRect);
		text.pos = textDir.align(textRect, textSize);
	}
}
