/*
 *   Copyright (C) 2008, 2009 Nicolas Vion <nico@yojik.eu>
 *
 *   This file is part of Swac-play.
 * 
 *   Swac-play is free software: you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, either version 3 of the License, or
 *   (at your option) any later version.
 *   
 *   Swac-play is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *   
 *   You should have received a copy of the GNU General Public License
 *   along with Swac-play.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <iostream>
#include <sstream>
#include <glibmm.h>

#include "sqlite3++.hh"
#include "gstreamer.hh"


bool db_check_version(SqlDatabase* db) {
	return true;
/*	SqlQuery query(db);
	query.prepare("PRAGMA user_version;");
	return (query.step() == SQLITE_ROW and query.columnInt(0) == 1); */
}

SqlDatabase* db_open() {
	Glib::ustring path(Glib::get_home_dir() + G_DIR_SEPARATOR_S ".swac" G_DIR_SEPARATOR_S "swac.db");
	
	if (Glib::file_test(path, Glib::FILE_TEST_EXISTS)) {
		SqlDatabase *db;  
	   	db = new class SqlDatabase;  
		db->open(path);

		if (db_check_version(db))
			return db;

		db->close();
		delete db;
		std::cerr << "Please upgrade the database structure version with \"swac-get\"." << std::endl;
		return NULL;
	}
	std::cerr << "Can not open the database. Please run \"swac-get\" to build it." << std::endl;
	return NULL;
}

Glib::ustring search(SqlDatabase *db, Glib::ustring filter, bool const explain=false) {
	SqlQuery query(db);
	Glib::ustring sql = 
		"SELECT packages.path, filename, sounds.idx, SWAC_TEXT, SWAC_LANG, SWAC_SPEAK_NAME, SWAC_HOMOGRAPHIDX FROM sounds "
		"INNER JOIN packages ON sounds.packages_idx = packages.idx "
		"WHERE " + filter + " "
		"ORDER BY SWAC_PRON_INTONATION DESC LIMIT 24;";

	if (explain)
		std::cout << "QUERY: " << sql << std::endl << "-----" <<std::endl;

	query.prepare(sql);
	Glib::ustring path;
	Glib::ustring result;
	while (query.step() == SQLITE_ROW) {
		path = query.columnText(0) + query.columnText(1);
		std::cout << query.columnInt(2) << ": " << query.columnText(3) << " (" << query.columnText(4) << ")" << "\t" << query.columnText(6) << " " << path << std::endl;
		if (result == "")
			result = path;
	}
	return result;
}

Glib::ustring from_int(int i) {
	std::ostringstream ssIn;
	ssIn << i;
	return ssIn.str();
}

int main(int argc, char **argv) {
	setlocale(LC_ALL, "");
	Glib::init();

	Glib::OptionContext options("<string to play>");
	options.set_summary("Swac-play is a simple command line program based on GStreamer which allows to play sounds from audio collections of words (SWAC). It search audio records from a SQLite3 database made by the \"swac-get\" program.");
	Glib::OptionGroup main_group("", "");
	options.set_main_group(main_group);
	
	Glib::ustring lang = "";
	Glib::OptionEntry entry_lang;
	entry_lang.set_short_name('l');
	entry_lang.set_long_name("lang");
	entry_lang.set_description("Search among records of a given language");
	entry_lang.set_arg_description("iso-639-3 code");
	main_group.add_entry(entry_lang, lang);

	Glib::ustring packid = "";
	Glib::OptionEntry entry_packid;
	entry_packid.set_short_name('p');
	entry_packid.set_long_name("packid");
	entry_packid.set_description("Search among records of a given package");
	entry_packid.set_arg_description("package identifier");
	main_group.add_entry(entry_packid, packid);
	
	Glib::ustring homographidx = "";
	Glib::OptionEntry entry_homographidx;
	entry_homographidx.set_short_name('h');
	entry_homographidx.set_long_name("homographidx");
	entry_homographidx.set_description("Search record with a given `SWAC_HOMOGRAPHIDX` value");
	entry_homographidx.set_arg_description("string");
	main_group.add_entry(entry_homographidx, homographidx);
	
	Glib::ustring intonation = "";
	Glib::OptionEntry entry_intonation;
	entry_intonation.set_short_name('n');
	entry_intonation.set_long_name("intonation");
	entry_intonation.set_description("Search record with a given `SWAC_PRON_INTONATION` value");
	entry_intonation.set_arg_description("string");
	main_group.add_entry(entry_intonation, intonation);

	int idx = -1;
	Glib::OptionEntry entry_idx;
	entry_idx.set_short_name('i');
	entry_idx.set_long_name("idx");
	entry_idx.set_description("Play sound with a given identifier");
	entry_idx.set_arg_description("integer");
	main_group.add_entry(entry_idx, idx);

	bool explain = false;
	Glib::OptionEntry entry_explain;
	entry_explain.set_short_name('e');
	entry_explain.set_long_name("explain");
	entry_explain.set_description("Show sql querie");
	main_group.add_entry(entry_explain, explain);

	bool version = false;
	Glib::OptionEntry entry_version;
	entry_version.set_short_name('v');
	entry_version.set_long_name("version");
	entry_version.set_description("Show the version number and exit");
	main_group.add_entry(entry_version, version);


	try {
		options.parse(argc, argv);
	}
	catch (const Glib::Exception& ex) {
		std::cerr << "Error: " << ex.what () << std::endl; 
		return 0;
	}

	if (version) {
		std::cout << "swac-play " VERSION << std::endl;
		return 0;
	}

	if (argc != 2 and idx == -1) {
		std::cerr << "You must specify a string" << std::endl; 
		return 0;
	}

	SqlDatabase *db;  
	if ((db = db_open()) != NULL) {
		Glib::ustring filter = (idx != -1)
			? "sounds.idx=" + from_int(idx)
			: "SWAC_TEXT=" + db->str(Glib::locale_to_utf8(argv[1]))
				+ ((lang != "")   ? " AND SWAC_LANG=" + db->str(lang) : "")
				+ ((homographidx != "")   ? " AND SWAC_HOMOGRAPHIDX=" + db->str(homographidx) : "")
				+ ((packid != "") ? " AND packages.packid=" + db->str(packid) : "")
				+ ((intonation != "") ? " AND SWAC_PRON_INTONATION=" + db->str(intonation) : "");


		Glib::ustring url = search(db, filter, explain);
		if (url != "") {
			std::cout << "Playing " << url << "..." << std::flush;
			Gst gst;
			gst.init(argc, argv);
			gst.play_url(url.c_str());
			gst.close();
			std::cout << "DONE." << std::endl;
		}
		else
			std::cout << "Can not find matching audio record." << std::endl;
		db->close();
		delete db;
	}

	return 0;
}
